<?php

namespace App\Services;

use App\Models\Transaction;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Route;
use Stripe\Stripe;
use Stripe\Checkout\Session as StripeSession;

class PaymentGatewayService
{
    public function initiatePayment($transaction, $gateway)
    {
        switch ($gateway->gateway_name) {
            case 'Stripe':
                return $this->handleStripePayment($transaction);
            case 'PayPal':
                return $this->handlePayPalPayment($transaction);
            case 'NowPayments':
                return $this->handleNowPaymentsPayment($transaction);
            case 'PerfectMoney':
                return $this->handlePerfectMoneyPayment($transaction);
            case 'Payeer':
                return $this->handlePayeerPayment($transaction);
            case 'Cryptocurrency':
                return $this->handleCryptoPayment($transaction);
            case 'ManualPayment':
                return $this->handleManualPayment($transaction);
            default:
                throw new \Exception('Unsupported payment gateway.');
        }
    }

    /**
     * Handle Stripe Payment
     *
     * @param Transaction $transaction
     * @return \Illuminate\Http\JsonResponse
     */
    private function handleStripePayment($transaction)
    {
        try {
            Stripe::setApiKey(env('STRIPE_SECRET_KEY'));

            $session = StripeSession::create([
                'payment_method_types' => ['card'],
                'line_items' => [[
                    'price_data' => [
                        'currency' => 'usd',
                        'product_data' => [
                            'name' => 'Deposit for Wallet',
                        ],
                        'unit_amount' => $transaction->amount * 100,
                    ],
                    'quantity' => 1,
                ]],
                'mode' => 'payment',
                'success_url' => route('wallet.payment.callback', ['transactionId' => $transaction->id]),
                'cancel_url' => route('wallet.index'),
                'client_reference_id' => $transaction->id,
            ]);

            Log::info("Stripe payment initiated successfully for Transaction: {$transaction->id}");

            return response()->json(['url' => $session->url]);
        } catch (\Exception $e) {
            Log::error("Stripe payment initiation error: " . $e->getMessage());
            throw new \Exception("Stripe payment initiation failed.");
        }
    }

    /**
     * Handle PayPal Payment
     *
     * @param Transaction $transaction
     * @return \Illuminate\Http\JsonResponse
     */
    private function handlePayPalPayment($transaction)
    {
        try {
            // PayPal payment logic (use PayPal SDK for real implementation)
            $paymentUrl = 'https://www.paypal.com/cgi-bin/webscr?...'; // Replace with real PayPal URL.
            Log::info("PayPal payment initiated for Transaction: {$transaction->id}");

            return response()->json(['url' => $paymentUrl]);
        } catch (\Exception $e) {
            Log::error("PayPal payment initiation error: " . $e->getMessage());
            throw new \Exception("PayPal payment initiation failed.");
        }
    }

    /**
     * Handle NowPayments Payment
     *
     * @param Transaction $transaction
     * @return \Illuminate\Http\JsonResponse
     */
    private function handleNowPaymentsPayment($transaction)
    {
        try {
            $response = Http::withHeaders([
                'x-api-key' => env('NOWPAYMENTS_API_KEY')
            ])->post('https://api.nowpayments.io/v1/payment', [
                'price_amount' => $transaction->amount,
                'price_currency' => 'usd',
                'order_id' => $transaction->id,
                'success_url' => route('wallet.payment.callback', ['transactionId' => $transaction->id]),
                'cancel_url' => route('wallet.index')
            ]);

            if ($response->successful()) {
                Log::info("NowPayments payment initiated successfully for Transaction: {$transaction->id}");
                return response()->json(['url' => $response->json('invoice_url')]);
            }

            throw new \Exception('NowPayments payment initiation failed.');
        } catch (\Exception $e) {
            Log::error("NowPayments payment initiation error: " . $e->getMessage());
            throw new \Exception("NowPayments payment initiation failed.");
        }
    }

    /**
     * Handle PerfectMoney Payment
     *
     * @param Transaction $transaction
     * @return \Illuminate\Http\JsonResponse
     */
    private function handlePerfectMoneyPayment($transaction)
    {
        try {
            // PerfectMoney payment initiation logic
            $paymentUrl = 'https://perfectmoney.is/api?...'; // Replace with the real PerfectMoney URL.
            Log::info("PerfectMoney payment initiated for Transaction: {$transaction->id}");

            return response()->json(['url' => $paymentUrl]);
        } catch (\Exception $e) {
            Log::error("PerfectMoney payment initiation error: " . $e->getMessage());
            throw new \Exception("PerfectMoney payment initiation failed.");
        }
    }

    /**
     * Handle Payeer Payment
     *
     * @param Transaction $transaction
     * @return \Illuminate\Http\JsonResponse
     */
    private function handlePayeerPayment($transaction)
    {
        try {
            // Payeer payment initiation logic
            $paymentUrl = 'https://payeer.com/api?...'; // Replace with the real Payeer URL.
            Log::info("Payeer payment initiated for Transaction: {$transaction->id}");

            return response()->json(['url' => $paymentUrl]);
        } catch (\Exception $e) {
            Log::error("Payeer payment initiation error: " . $e->getMessage());
            throw new \Exception("Payeer payment initiation failed.");
        }
    }

    /**
     * Handle Cryptocurrency Payment
     *
     * @param Transaction $transaction
     * @return \Illuminate\Http\JsonResponse
     */
    private function handleCryptoPayment($transaction)
    {
        try {
            // Cryptocurrency payment logic
            $paymentUrl = 'https://cryptopaymentgateway.com/...'; // Replace with the real URL for crypto payments.
            Log::info("Cryptocurrency payment initiated for Transaction: {$transaction->id}");

            return response()->json(['url' => $paymentUrl]);
        } catch (\Exception $e) {
            Log::error("Cryptocurrency payment initiation error: " . $e->getMessage());
            throw new \Exception("Cryptocurrency payment initiation failed.");
        }
    }

    /**
     * Handle Manual Payment
     *
     * @param Transaction $transaction
     * @return \Illuminate\Http\JsonResponse
     */
    private function handleManualPayment($transaction)
    {
        try {
            Log::info("Manual payment instructions requested for Transaction: {$transaction->id}");

            return response()->json(['url' => route('wallet.index'), 'message' => 'Please contact support for manual payment instructions.']);
        } catch (\Exception $e) {
            Log::error("Manual payment handling error: " . $e->getMessage());
            throw new \Exception("Manual payment handling failed.");
        }
    }
}
