<?php

namespace App\Services;

use Exception;
use Illuminate\Support\Facades\DB;

class MT5ManagerService
{
    private $connection;
    private $server;
    private $port;
    private $managerLogin;
    private $managerPassword;

    /**
     * MT5ManagerService constructor.
     * Initializes connection parameters from the database.
     *
     * @param string $serverType ('real' or 'demo')
     * @throws Exception
     */
    public function __construct($serverType = 'real')
    {
        $settings = DB::table('mt5_settings')->where('server_type', $serverType)->first();

        if (!$settings) {
            throw new Exception("MT5 settings not configured for $serverType server.");
        }

        $this->server = $settings->server;
        $this->port = $settings->port;
        $this->managerLogin = $settings->manager_login;
        $this->managerPassword = $settings->manager_password;
    }

    /**
     * Connect to the MT5 server.
     *
     * @throws Exception
     */
    private function connect()
    {
        // Custom connection logic (replace with your implementation)
        $this->connection = new MT5Connection($this->server, $this->port, $this->managerLogin, $this->managerPassword);

        if (!$this->connection->connect()) {
            throw new Exception("Failed to connect to MT5 server.");
        }
    }

    /**
     * Disconnect from the MT5 server.
     */
    public function disconnect()
    {
        if ($this->connection) {
            $this->connection->disconnect();
        }
    }

    /**
     * Create a new MT5 account.
     *
     * @param array $data
     * @return array
     */
    public function createAccount(array $data)
    {
        $this->connect();

        try {
            $response = $this->connection->createAccount($data);

            if ($response['status'] === 'success') {
                return [
                    'status' => 'success',
                    'data' => $response['data'],
                ];
            }

            return [
                'status' => 'error',
                'message' => $response['message'],
            ];
        } catch (Exception $e) {
            return $this->handleException($e);
        } finally {
            $this->disconnect();
        }
    }

    public function getAllAccounts()
    {
        $this->connect();

        try {
            $accounts = $this->connection->getAllAccounts();
            return [
                'status' => 'success',
                'data' => $accounts,
            ];
        } catch (Exception $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage(),
            ];
        } finally {
            $this->disconnect();
        }
    }

    public function deposit($login, $amount, $comment = 'Deposit')
    {
        $this->connect();

        try {
            // Simulating the deposit operation
            $trade = [
                'login' => $login,
                'amount' => $amount,
                'comment' => $comment,
                'type' => 'credit', // Adjust this based on MT5 implementation
            ];

            // Placeholder for the actual API call to perform the deposit
            $response = $this->connection->transferFunds($login, $amount, $comment, 'deposit');

            return [
                'status' => 'success',
                'data' => $trade,
                'message' => 'Deposit successful.',
            ];
        } catch (\Exception $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage(),
            ];
        } finally {
            $this->disconnect();
        }
    }


    public function transferFunds($login, $amount, $comment, $action)
    {
        $this->connect();

        try {
            $type = ($action === 'deposit') ? 'credit' : 'debit'; // Define based on action
            $trade = [
                'login' => $login,
                'amount' => $amount,
                'comment' => $comment,
                'type' => $type,
            ];

            // Simulate API call or actual implementation
            $response = $this->connection->executeTransfer($trade);

            return [
                'status' => 'success',
                'data' => $trade,
            ];
        } catch (\Exception $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage(),
            ];
        } finally {
            $this->disconnect();
        }
    }


    /**
     * Withdraw funds from an MT5 account.
     *
     * @param string $login
     * @param float $amount
     * @param string $comment
     * @return array
     */
    public function withdraw($login, $amount, $comment = 'Withdrawal')
    {
        $this->connect();

        try {
            $response = $this->connection->transferFunds($login, -1 * abs($amount), $comment, 'withdrawal');

            if ($response['status'] === 'success') {
                return [
                    'status' => 'success',
                    'message' => 'Withdrawal successful.',
                ];
            }

            return [
                'status' => 'error',
                'message' => $response['message'],
            ];
        } catch (Exception $e) {
            return $this->handleException($e);
        } finally {
            $this->disconnect();
        }
    }

    /**
     * Fetch trades for a specific account.
     *
     * @param string $login
     * @return array
     */
    public function getTrades($login)
    {
        $this->connect();

        try {
            $trades = $this->connection->getTrades($login);

            return [
                'status' => 'success',
                'data' => $trades,
            ];
        } catch (Exception $e) {
            return $this->handleException($e);
        } finally {
            $this->disconnect();
        }
    }

    /**
     * Fetch all trades.
     *
     * @return array
     */
    public function getAllTrades()
    {
        $this->connect();

        try {
            $trades = $this->connection->getAllTrades();

            return [
                'status' => 'success',
                'data' => $trades,
            ];
        } catch (Exception $e) {
            return $this->handleException($e);
        } finally {
            $this->disconnect();
        }
    }

    /**
     * Handle exceptions and return structured error messages.
     *
     * @param Exception $e
     * @return array
     */
    private function handleException(Exception $e)
    {
        return [
            'status' => 'error',
            'message' => $e->getMessage(),
        ];
    }
}
