<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'referred_by', // Referring user ID (IB or other user)
        'is_active',   // Active status of the user
        'is_ib',       // Boolean indicating if the user is an IB
        'mt5_account_id',
        'mt5_demo_account_id'

    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
    ];

    /**
     * Relationships
     */
    // Direct referrals using the Referral model (for earnings tracking)
    public function directReferrals()
    {
        return $this->hasMany(Referral::class, 'referrer_id');
    }


    /**
     * Relationship: Parent IB through the IBHierarchy table.
     */
    public function parentIB()
    {
        return $this->hasOneThrough(
            self::class,
            IBHierarchy::class,
            'ib_id',       // Foreign key on IBHierarchy table
            'id',          // Foreign key on User table
            'id',          // Local key on User table
            'parent_id'    // Local key on IBHierarchy table
        );
    }

    /**
     * Relationship: Referred users (child IBs) through the IBHierarchy table.
     */
    public function referredUsers()
    {
        return $this->hasManyThrough(
            self::class,
            IBHierarchy::class,
            'parent_id',    // Foreign key on IBHierarchy table
            'id',           // Foreign key on User table
            'id',           // Local key on User table
            'ib_id'         // Local key on IBHierarchy table
        );
    }




    /**
     * Relationship: IB Hierarchy
     */
    public function ibHierarchy()
    {
        return $this->hasMany(IBHierarchy::class, 'ib_id');
    }

    // User who referred this user
    public function referrer()
    {
        return $this->belongsTo(User::class, 'referred_by');
    }


    /**
     * Relationship: IB Relationships
     */
    public function ibRelationships()
    {
        return $this->hasMany(IBRelationship::class, 'ib_id');
    }
    /**
     * Relationship: Referrals
     */
    public function referrals()
    {
        return $this->hasMany(Referral::class, 'referrer_id');
    }

    // The IB who referred this user
    public function referredByIB()
    {
        return $this->belongsTo(IBRelationship::class, 'referred_user_id');
    }

    // Trades associated with the user
    public function trades()
    {
        return $this->hasMany(Trade::class);
    }

    // Wallet associated with the user
    public function wallet()
    {
        return $this->hasOne(Wallet::class);
    }

    // Roles assigned to the user (many-to-many)
    public function roles()
    {
        return $this->belongsToMany(Role::class, 'role_user', 'user_id', 'role_id')->withTimestamps();
    }

    // Transactions associated with the user
    public function transactions()
    {
        return $this->hasMany(Transaction::class);
    }

    // Activity logs for the user
    public function activityLogs()
    {
        return $this->hasMany(ActivityLog::class);
    }

    // Trading accounts associated with the user
    public function tradingAccounts()
    {
        return $this->hasMany(TradingAccount::class);
    }

    /**
     * Role Management Functions
     */

    // Check if the user has a specific role
    public function hasRole(string $role): bool
    {
        return $this->roles->pluck('name')->contains($role);
    }

    // Assign a role to the user
    public function assignRole($role): bool
    {
        $roleId = is_string($role) ? Role::where('name', $role)->value('id') : $role;
        if ($roleId && !$this->roles->pluck('id')->contains($roleId)) {
            $this->roles()->attach($roleId);
            return true;
        }
        return false;
    }

    // Remove a role from the user
    public function removeRole($role): bool
    {
        $roleId = is_string($role) ? Role::where('name', $role)->value('id') : $role;
        if ($roleId && $this->roles->pluck('id')->contains($roleId)) {
            $this->roles()->detach($roleId);
            return true;
        }
        return false;
    }

    // Check if the user has any of the specified roles
    public function hasAnyRole(array $roles): bool
    {
        return $this->roles->pluck('name')->intersect($roles)->isNotEmpty();
    }

    // Check if the user has all of the specified roles
    public function hasAllRoles(array $roles): bool
    {
        return $this->roles->pluck('name')->intersect($roles)->count() === count($roles);
    }

    /**
     * Utility Functions
     */

    // Check if the user is an IB
    public function isIB(): bool
    {
        return $this->is_ib;
    }

    // Get the count of referrals for this user
    public function getReferralsCount(): int
    {
        return $this->referrals()->count();
    }

    // Calculate total earnings for the user
    public function calculateTotalEarnings(): float
    {
        return $this->trades->sum('profit'); // Assuming there's a `profit` field in `trades`
    }



    /**
     * Accessors
     */

    // Get user's full name in uppercase
    public function getFullNameAttribute(): string
    {
        return strtoupper($this->name);
    }

    // Get user's account status as text
    public function getAccountStatusAttribute(): string
    {
        return $this->is_active ? 'Active' : 'Inactive';
    }

    // Get formatted creation date of the user
    public function getFormattedCreatedAtAttribute(): string
    {
        return $this->created_at->format('d M, Y H:i');
    }

    /**
     * Scopes
     */

    // Scope to filter active users
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    // Scope to filter inactive users
    public function scopeInactive($query)
    {
        return $query->where('is_active', false);
    }

    // Scope to filter IB users
    public function scopeIB($query)
    {
        return $query->where('is_ib', true);
    }
}
