<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;

class Transaction extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'type',
        'amount',
        'status',
        'payment_gateway',
        'gateway_transaction_id',
        'admin_id',
        'updated_status_at',
        'currency',
        'description'
    ];

    /**
     * Relationships
     */

    // Belongs to a User
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    // Belongs to a Wallet
    public function wallet()
    {
        return $this->belongsTo(Wallet::class);
    }

    // Belongs to an Admin (when an admin approves or rejects transactions)
    public function admin()
    {
        return $this->belongsTo(User::class, 'admin_id');
    }

    /**
     * Accessors and Mutators
     */

    // Accessor for formatted amount
    public function getFormattedAmountAttribute()
    {
        return '$' . number_format($this->amount, 2);
    }

    // Accessor for formatted date
    public function getFormattedDateAttribute()
    {
        return $this->created_at->format('d M Y, h:i A');
    }

    /**
     * Scopes
     */

    // Scope to filter transactions by status
    public function scopeStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    // Scope to filter transactions by type (deposit or withdrawal)
    public function scopeType($query, $type)
    {
        return $query->where('type', $type);
    }

    // Scope to filter transactions by date range
    public function scopeDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('created_at', [$startDate, $endDate]);
    }

    /**
     * Boot Method for Additional Logic
     */
    protected static function boot()
    {
        parent::boot();

        // Automatically set `updated_status_at` when the status is updated
        static::updating(function ($transaction) {
            if ($transaction->isDirty('status')) {
                $transaction->updated_status_at = now();
                $transaction->admin_id = Auth::id(); // Set the admin ID if available
            }
        });
    }
}
