<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Services\MT5ManagerService;
use Illuminate\Support\Facades\Auth;

class UserMT5Controller extends Controller
{
    protected $mt5Service;

    /**
     * Inject the MT5ManagerService.
     */
    public function __construct(MT5ManagerService $mt5Service)
    {
        $this->mt5Service = $mt5Service;
    }

    /**
     * Create a live MT5 account.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function createLiveAccount(Request $request)
    {
        $request->validate([
            'leverage' => 'required|integer|min:50|max:500',
            'currency' => 'required|string|max:10',
            'initial_balance' => 'required|numeric|min:100',
            'group' => 'required|string|max:50',
        ]);

        $user = Auth::user();

        $response = $this->mt5Service->createAccount([
            'name' => $user->name,
            'group' => $request->group,
            'leverage' => $request->leverage,
            'main_password' => 'UserMainPassword123', // Auto-generate password
            'investor_password' => 'InvestorPass123',
            'phone_password' => 'PhonePass123',
            'currency' => $request->currency,
            'initial_balance' => $request->initial_balance,
        ]);

        if ($response['status'] === 'success') {
            return redirect()->back()->with('success', 'Live MT5 account created successfully!');
        }

        return redirect()->back()->with('error', $response['message'] ?? 'Failed to create live account.');
    }

    /**
     * Create a demo MT5 account.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function createDemoAccount(Request $request)
    {
        $request->validate([
            'leverage' => 'required|integer|min:50|max:500',
            'currency' => 'required|string|max:10',
            'initial_balance' => 'required|numeric|min:100',
            'group' => 'required|string|max:50',
        ]);

        $user = Auth::user();

        $response = $this->mt5Service->createAccount([
            'name' => $user->name,
            'group' => $request->group,
            'leverage' => $request->leverage,
            'main_password' => 'UserMainPassword123',
            'investor_password' => 'InvestorPass123',
            'phone_password' => 'PhonePass123',
            'currency' => $request->currency,
            'initial_balance' => $request->initial_balance,
        ]);

        if ($response['status'] === 'success') {
            return redirect()->back()->with('success', 'Demo MT5 account created successfully!');
        }

        return redirect()->back()->with('error', $response['message'] ?? 'Failed to create demo account.');
    }

    /**
     * Fetch MT5 trades for the authenticated user.
     *
     * @return \Illuminate\View\View|\Illuminate\Http\RedirectResponse
     */
    public function fetchTrades()
    {
        $user = Auth::user();

        // Ensure the user has an MT5 account ID
        if (empty($user->mt5_account_id)) {
            return redirect()->back()->with('error', 'You do not have an MT5 account linked to your profile.');
        }

        $response = $this->mt5Service->getTrades($user->mt5_account_id);

        if ($response['status'] === 'success') {
            $trades = $response['data'];
            return view('user.mt5.trades', compact('trades'));
        }

        return redirect()->back()->with('error', $response['message'] ?? 'Failed to fetch trades.');
    }
}
