<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Trade;
use App\Models\ActivityLog;
use App\Models\Referral;
use App\Models\IBCommissionLevel;
use App\Models\User;

class TradeController extends Controller
{
    /**
     * Display a listing of the user's trades.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $trades = Trade::where('user_id', Auth::id())->latest()->paginate(10);

        return view('user.trades', compact('trades'));
    }

    /**
     * Open a new trade.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function openTrade(Request $request)
    {
        $request->validate([
            'asset' => 'required|string|max:255',
            'amount' => 'required|numeric|min:10',
            'type' => 'required|in:buy,sell',
        ]);

        // Create the new trade
        $trade = Trade::create([
            'user_id' => Auth::id(),
            'asset' => $request->asset,
            'amount' => $request->amount,
            'type' => $request->type,
            'status' => 'open',
        ]);

        // Log trade opening
        ActivityLog::create([
            'user_id' => Auth::id(),
            'activity_type' => 'trade_open',
            'details' => "Opened a trade: {$trade->type} {$trade->asset} with amount {$trade->amount}",
        ]);

        // Calculate IB Earnings for the referrer if applicable
        $this->calculateIBEarnings(Auth::id(), $request->amount);

        return back()->with('success', 'Trade opened successfully.');
    }

    /**
     * Close an existing trade.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function closeTrade(Request $request, $id)
    {
        $trade = Trade::where('user_id', Auth::id())->where('id', $id)->firstOrFail();

        if ($trade->status !== 'open') {
            return back()->with('error', 'Trade is not open.');
        }

        // Update trade status to closed and set profit/loss
        $trade->status = 'closed';
        $trade->profit_loss = $request->input('profit_loss', 0);
        $trade->save();

        // Log trade closing
        ActivityLog::create([
            'user_id' => Auth::id(),
            'activity_type' => 'trade_close',
            'details' => "Closed trade: {$trade->type} {$trade->asset} with profit/loss {$trade->profit_loss}",
        ]);

        // Recalculate IB Earnings based on profit/loss if applicable
        $this->calculateIBEarnings(Auth::id(), $trade->profit_loss, true);

        return back()->with('success', 'Trade closed successfully.');
    }

    /**
     * Calculate IB earnings for the referrer.
     *
     * @param int $userId
     * @param float $tradeAmount
     * @param bool $isClosing
     * @return void
     */
    private function calculateIBEarnings(int $userId, float $tradeAmount, bool $isClosing = false)
    {
        // Get the user making the trade
        $user = User::findOrFail($userId);

        // Get the IB referrer of the user
        $referrer = $user->referrer;

        if ($referrer && $referrer->is_ib) {
            // Retrieve commission level for this IB's level
            $commissionLevel = IBCommissionLevel::where('level', $referrer->level)->first();
            $commissionPercentage = $commissionLevel ? $commissionLevel->commission_percentage : 0;

            // Calculate earnings
            $earnings = $tradeAmount * ($commissionPercentage / 100);

            if ($isClosing) {
                // Adjust earnings calculation if it's for a closing trade (using profit/loss)
                $earnings = $earnings < 0 ? 0 : $earnings; // Ensure that earnings cannot be negative
            }

            // Update or create referral record
            Referral::updateOrCreate(
                ['referrer_id' => $referrer->id, 'referred_user_id' => $userId],
                [
                    'earnings' => $earnings,
                    'bonus' => 0, // Adjust as per your logic for any bonus if applicable
                ]
            );
        }
    }
}
