<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Role;
use App\Models\Permission;
use App\Models\User;

class PermissionController extends Controller
{
    /**
     * Display a list of roles and their permissions.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $roles = Role::with('permissions')->get();
        $permissions = Permission::all();

        return view('admin.permissions.index', compact('roles', 'permissions'));
    }

    /**
     * Store a new role or permission.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|unique:' . ($request->type === 'role' ? 'roles' : 'permissions') . ',name',
            'type' => 'required|in:role,permission',
        ]);

        if ($request->type === 'role') {
            Role::create(['name' => $request->name]);
        } else {
            Permission::create(['name' => $request->name]);
        }

        return redirect()->back()->with('success', ucfirst($request->type) . ' created successfully.');
    }

    /**
     * Assign permissions to a role.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function assignPermissionsToRole(Request $request)
    {
        $request->validate([
            'role_id' => 'required|exists:roles,id',
            'permissions' => 'array',
            'permissions.*' => 'exists:permissions,id',
        ]);

        $role = Role::findOrFail($request->role_id);
        $role->permissions()->sync($request->permissions);

        return redirect()->back()->with('success', 'Permissions assigned to role successfully.');
    }

    /**
     * Assign roles to a user.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function assignRolesToUser(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'roles' => 'array',
            'roles.*' => 'exists:roles,id',
        ]);

        $user = User::findOrFail($request->user_id);
        $user->roles()->sync($request->roles);

        return redirect()->back()->with('success', 'Roles assigned to user successfully.');
    }

    /**
     * Display form for creating roles or permissions.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $roles = Role::all();
        $permissions = Permission::all();
        return view('admin.permissions.create', compact('roles', 'permissions'));
    }
}
