<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Role;
use App\Models\Referral;
use App\Providers\RouteServiceProvider;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;
use Illuminate\View\View;

class RegisteredUserController extends Controller
{
    /**
     * Display the registration view.
     */
    public function create(): View
    {
        return view('auth.register');
    }

    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        // Validate the incoming registration request
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:' . User::class],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
        ]);

        // Create the new user
        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
        ]);

        // Assign the default "user" role to the new user
        $userRole = Role::where('name', 'user')->first();
        if ($userRole) {
            $user->roles()->attach($userRole->id);
        }

        // Handle referral logic
        $this->handleReferral($user);

        // Trigger the registered event
        event(new Registered($user));

        // Log in the new user
        Auth::login($user);

        // Redirect to the home route
        return redirect(RouteServiceProvider::HOME);
    }

    /**
     * Handle referral system logic.
     *
     * @param \App\Models\User $user
     * @return void
     */
    private function handleReferral(User $user)
    {
        $referrerId = session()->get('referrer_id'); // Get the referrer ID from the session
        if ($referrerId) {
            Referral::create([
                'referrer_id' => $referrerId,
                'referred_user_id' => $user->id,
            ]);

            // Clear the referrer ID from the session after use
            session()->forget('referrer_id');
        }
    }
}
