<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Referral;
use App\Models\Role;
use Illuminate\Foundation\Auth\RegistersUsers;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Http\Request;
use App\Mail\UserNotificationMail;
use Illuminate\Support\Facades\Mail;

class RegisterController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Register Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users as well as their
    | validation and creation. By default, this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
    */

    use RegistersUsers;

    /**
     * Where to redirect users after registration.
     *
     * @var string
     */
    protected $redirectTo = '/home';

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest');
    }

    /**
     * Show the registration form and store referrer ID if available.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Contracts\View\View
     */
    public function showRegistrationForm(Request $request)
    {
        if ($request->has('ref')) {
            session()->put('referrer_id', $request->get('ref'));
        }

        return view('auth.register');
    }

    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array  $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function validator(array $data)
    {
        return Validator::make($data, [
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email'], // Ensure users table is referenced
            'password' => ['required', 'string', 'min:8', 'confirmed'],
        ]);
    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     * @return \App\Models\User
     */
    protected function create(array $data)
    {
        // Create the user
        $user = User::create([
            'name' => $data['name'],
            'email' => $data['email'],
            'password' => Hash::make($data['password']),
        ]);

        // Assign the default "user" role to the new user
        $this->assignDefaultRole($user);

        // Handle referral logic
        $this->handleReferral($user);

        return $user;
    }

    /**
     * Handle the referral system logic.
     *
     * @param \App\Models\User $user
     * @return void
     */
    protected function handleReferral(User $user)
    {
        $referrerId = session()->get('referrer_id'); // Retrieve the referrer ID from the session
        if ($referrerId) {
            Referral::create([
                'referrer_id' => $referrerId,
                'referred_user_id' => $user->id,
            ]);

            // Optionally, clear the referrer ID from the session after use
            session()->forget('referrer_id');
        }
    }


    /**
     * Assign the default role to a newly registered user.
     *
     * @param \App\Models\User $user
     * @return void
     */
    protected function assignDefaultRole(User $user)
    {
        $defaultRole = Role::where('name', 'user')->first();
        if ($defaultRole) {
            $user->roles()->attach($defaultRole->id);
        }
    }
}
