<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Services\MT5ManagerService;
use Illuminate\Support\Facades\DB;
use Exception;

class AdminMT5Controller extends Controller
{
    protected $mt5Service;

    /**
     * Constructor to initialize the MT5ManagerService.
     *
     * @param MT5ManagerService $mt5Service
     */
    public function __construct(MT5ManagerService $mt5Service)
    {
        $this->mt5Service = $mt5Service;
    }

    /**
     * List all MT5 accounts.
     *
     * @return \Illuminate\View\View
     */
    public function listAccounts()
    {
        try {
            $response = $this->mt5Service->getAllAccounts();

            if ($response['status'] === 'success') {
                $accounts = $response['data'];
            } else {
                $accounts = [];
                session()->flash('error', $response['message'] ?? 'Failed to fetch accounts.');
            }
        } catch (Exception $e) {
            $accounts = [];
            session()->flash('error', 'An error occurred while fetching accounts: ' . $e->getMessage());
        }

        return view('admin.mt5_accounts', compact('accounts'));
    }

    /**
     * Update MT5 settings (Real or Demo server).
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request)
    {
        $request->validate([
            'server_type' => 'required|in:real,demo',
            'server' => 'required|string|max:255',
            'port' => 'required|integer|min:1|max:65535',
            'manager_login' => 'required|string|max:255',
            'manager_password' => 'required|string|max:255',
        ]);

        try {
            DB::table('mt5_settings')->updateOrInsert(
                ['server_type' => $request->server_type],
                [
                    'server' => $request->server,
                    'port' => $request->port,
                    'manager_login' => $request->manager_login,
                    'manager_password' => $request->manager_password,
                    'updated_at' => now(),
                ]
            );

            return back()->with('success', ucfirst($request->server_type) . ' MT5 settings updated successfully.');
        } catch (Exception $e) {
            return back()->with('error', 'Failed to update MT5 settings: ' . $e->getMessage());
        }
    }

    /**
     * View all trades across MT5 accounts.
     *
     * @return \Illuminate\View\View
     */
    public function viewAllTrades()
    {
        try {
            $response = $this->mt5Service->getAllTrades();

            if ($response['status'] === 'success') {
                $trades = $response['data'];
            } else {
                $trades = [];
                session()->flash('error', $response['message'] ?? 'Failed to fetch trades.');
            }
        } catch (Exception $e) {
            $trades = [];
            session()->flash('error', 'An error occurred while fetching trades: ' . $e->getMessage());
        }

        return view('admin.trades', compact('trades'));
    }

    /**
     * View MT5 settings (Real and Demo servers).
     *
     * @return \Illuminate\View\View
     */
    public function viewMT5Settings()
    {
        $realSettings = DB::table('mt5_settings')->where('server_type', 'real')->first();
        $demoSettings = DB::table('mt5_settings')->where('server_type', 'demo')->first();

        return view('admin.mt5_settings', compact('realSettings', 'demoSettings'));
    }

    /**
     * Create an MT5 account.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function createAccount(Request $request)
    {
        $request->validate([
            'name' => 'required|string',
            'group' => 'required|string',
            'leverage' => 'required|integer',
            'main_password' => 'required|string',
            'investor_password' => 'required|string',
            'phone_password' => 'required|string',
            'server_type' => 'required|in:real,demo',
        ]);

        try {
            $mt5Service = new MT5ManagerService($request->server_type);
            $response = $mt5Service->createAccount($request->all());

            if ($response['status'] === 'error') {
                return back()->with('error', $response['message']);
            }

            return back()->with('success', 'MT5 account created successfully!');
        } catch (Exception $e) {
            return back()->with('error', 'Failed to create MT5 account: ' . $e->getMessage());
        }
    }
    /**
     * Deposit funds into an MT5 account.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function deposit(Request $request)
    {
        $request->validate([
            'login' => 'required|string',
            'amount' => 'required|numeric|min:1',
            'comment' => 'nullable|string|max:255',
        ]);

        try {
            $response = $this->mt5Service->deposit(
                $request->login,
                $request->amount,
                $request->comment ?? 'Admin Deposit'
            );

            if ($response['status'] === 'success') {
                return redirect()->back()->with('success', 'Funds deposited successfully.');
            }

            return redirect()->back()->with('error', $response['message'] ?? 'Failed to deposit funds.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }


    /**
     * Withdraw funds from an MT5 account.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function withdraw(Request $request)
    {
        $request->validate([
            'login' => 'required|integer',
            'amount' => 'required|numeric|min:0.01',
            'server_type' => 'required|in:real,demo',
        ]);

        try {
            $mt5Service = new MT5ManagerService($request->server_type);
            $response = $mt5Service->withdraw($request->login, $request->amount);

            if ($response['status'] === 'error') {
                return back()->with('error', $response['message']);
            }

            return back()->with('success', 'Funds withdrawn successfully!');
        } catch (Exception $e) {
            return back()->with('error', 'Failed to withdraw funds: ' . $e->getMessage());
        }
    }
}
