<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Transaction;
use App\Models\ActivityLog;
use Illuminate\Support\Facades\DB;

class AdminDepositController extends Controller
{
    /**
     * Display a list of pending deposit requests.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $deposits = Transaction::where('type', 'deposit')
            ->where('status', 'pending')
            ->latest()
            ->paginate(10); // Change this to paginate results

        // Count the total approved and rejected deposits for summary cards
        $approvedDepositsCount = Transaction::where('type', 'deposit')
            ->where('status', 'approved')
            ->count();

        $rejectedDepositsCount = Transaction::where('type', 'deposit')
            ->where('status', 'rejected')
            ->count();

        return view('admin.deposits.index', compact('deposits', 'approvedDepositsCount', 'rejectedDepositsCount'));
    }
    /**
     * Approve a pending deposit request.
     *
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function approve($id)
    {
        $transaction = Transaction::findOrFail($id);

        if ($transaction->status !== 'pending') {
            return back()->with('error', 'Deposit request is not in pending state.');
        }

        DB::transaction(function () use ($transaction) {
            // Mark transaction as approved
            $transaction->status = 'approved';
            $transaction->save();

            // Update wallet balance
            $wallet = $transaction->user->wallet;
            $wallet->balance += $transaction->amount;
            $wallet->save();

            // Mark the transaction as completed
            $transaction->status = 'completed';
            $transaction->save();

            // Log deposit approval activity
            ActivityLog::create([
                'user_id' => $transaction->user_id,
                'activity_type' => 'deposit_approved',
                'details' => "Approved deposit for amount: " . $transaction->amount,
                'ip_address' => request()->ip(),
            ]);
        });

        return back()->with('success', 'Deposit approved successfully.');
    }

    /**
     * Reject a pending deposit request.
     *
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function reject($id)
    {
        $transaction = Transaction::findOrFail($id);

        if ($transaction->status !== 'pending') {
            return back()->with('error', 'Deposit request is not in pending state.');
        }

        // Mark transaction as rejected
        $transaction->status = 'rejected';
        $transaction->save();

        // Log deposit rejection activity
        ActivityLog::create([
            'user_id' => $transaction->user_id,
            'activity_type' => 'deposit_rejected',
            'details' => "Rejected deposit for amount: " . $transaction->amount,
            'ip_address' => request()->ip(),
        ]);

        return back()->with('success', 'Deposit rejected successfully.');
    }
}
