<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Referral;
use App\Models\User;
use App\Models\IBCommissionLevel;
use Illuminate\Support\Facades\DB;

class IBEarningsController extends Controller
{
    /**
     * Display the IB Earnings report with advanced filters and sorting.
     *
     * @param Request $request
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        // Fetch filters
        $search = $request->input('search');
        $level = $request->input('level');
        $status = $request->input('status');
        $sort = $request->input('sort', 'latest');

        // Build the base query with relations for referrer and referred user
        $query = Referral::with(['referrer', 'referredUser'])
            ->where('earnings', '>', 0);

        // Apply search filter by referrer or referred user name/email
        if (!empty($search)) {
            $query->whereHas('referrer', function ($q) use ($search) {
                $q->where('name', 'like', '%' . $search . '%')
                    ->orWhere('email', 'like', '%' . $search . '%');
            })->orWhereHas('referredUser', function ($q) use ($search) {
                $q->where('name', 'like', '%' . $search . '%')
                    ->orWhere('email', 'like', '%' . $search . '%');
            });
        }

        // Apply filter by IB level if provided
        if (!empty($level)) {
            $query->whereHas('referrer', function ($q) use ($level) {
                $q->whereHas('ibHierarchy', function ($subQuery) use ($level) {
                    $subQuery->where('level', $level);
                });
            });
        }

        // Apply filter by status (active/inactive)
        if (!empty($status)) {
            $isActive = $status === 'active';
            $query->whereHas('referredUser', function ($q) use ($isActive) {
                $q->where('is_active', $isActive);
            });
        }

        // Apply sorting
        if ($sort === 'latest') {
            $query->latest();
        } elseif ($sort === 'oldest') {
            $query->oldest();
        } elseif ($sort === 'highest_earning') {
            $query->orderBy('earnings', 'desc');
        } elseif ($sort === 'lowest_earning') {
            $query->orderBy('earnings', 'asc');
        }

        // Fetch earnings data with pagination
        $earnings = $query->paginate(10);

        // Calculate additional metrics
        $totalEarnings = $query->sum('earnings');
        $totalActiveEarnings = Referral::whereHas('referredUser', function ($q) {
            $q->where('is_active', true);
        })->sum('earnings');
        $totalInactiveEarnings = Referral::whereHas('referredUser', function ($q) {
            $q->where('is_active', false);
        })->sum('earnings');
        $totalIBs = User::where('is_ib', true)->count();
        $totalActiveReferrals = Referral::whereHas('referredUser', function ($q) {
            $q->where('is_active', true);
        })->count();
        $totalInactiveReferrals = Referral::whereHas('referredUser', function ($q) {
            $q->where('is_active', false);
        })->count();

        // Get list of IB commission levels for filtering purposes
        $commissionLevels = IBCommissionLevel::all();

        return view('admin.ib-earnings', compact(
            'earnings',
            'totalEarnings',
            'totalActiveEarnings',
            'totalInactiveEarnings',
            'totalIBs',
            'totalActiveReferrals',
            'totalInactiveReferrals',
            'commissionLevels',
            'search',
            'level',
            'status',
            'sort'
        ));
    }


    /**
     * Update the IB earnings based on commission levels.
     *
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateEarnings(Request $request, $id)
    {
        // Validate request data
        $request->validate([
            'trade_amount' => 'required|numeric|min:0',
        ]);

        // Find the referral and related IB user
        $referral = Referral::findOrFail($id);
        $ib = User::findOrFail($referral->referrer_id);

        // Retrieve commission level for this IB's level
        $commissionLevel = IBCommissionLevel::where('level', $ib->ibHierarchy->level ?? 0)->first();
        $commissionPercentage = $commissionLevel ? $commissionLevel->commission_percentage : 0;

        // Calculate new earnings based on the provided trade amount
        $earnings = $request->trade_amount * ($commissionPercentage / 100);

        // Update the referral earnings with rounded value
        $referral->earnings = round($earnings, 2);
        $referral->save();

        return redirect()->route('admin.ib.earnings')->with('success', 'IB earnings updated successfully.');
    }

    /**
     * Display details of a specific IB earning.
     *
     * @param int $id
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        $referral = Referral::with(['referrer', 'referredUser'])->findOrFail($id);

        return view('admin.ib-earnings-detail', compact('referral'));
    }

    /**
     * Recalculate earnings for all IBs.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function recalculateEarnings()
    {
        DB::transaction(function () {
            $referrals = Referral::all();

            foreach ($referrals as $referral) {
                $ib = User::find($referral->referrer_id);
                if ($ib && $ib->ibHierarchy) {
                    $commissionLevel = IBCommissionLevel::where('level', $ib->ibHierarchy->level)->first();
                    $commissionPercentage = $commissionLevel ? $commissionLevel->commission_percentage : 0;
                    $tradeAmount = 1000; // Example trade amount, adjust as needed
                    $earnings = $tradeAmount * ($commissionPercentage / 100);
                    $referral->earnings = round($earnings, 2);
                    $referral->save();
                }
            }
        });

        return redirect()->route('admin.ib.earnings')->with('success', 'IB earnings recalculated successfully.');
    }
}
