<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\ShouldAutoSize; // Optional to auto-size columns
use Maatwebsite\Excel\Concerns\WithMapping;    // Optional to map data in a custom format
use Maatwebsite\Excel\Concerns\WithStyles;     // Optional to apply styles to the Excel sheet
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use App\Models\Transaction;

class TransactionsExport implements FromCollection, WithHeadings, ShouldAutoSize, WithMapping, WithStyles
{
    /**
     * Return the collection of transactions to be exported.
     *
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return Transaction::with('user') // Load related user data
            ->select('id', 'user_id', 'amount', 'type', 'status', 'created_at')
            ->get();
    }

    /**
     * Return the headings for the columns in the exported file.
     *
     * @return array
     */
    public function headings(): array
    {
        return [
            'ID',
            'User Name',
            'Amount',
            'Type',
            'Status',
            'Date',
        ];
    }

    /**
     * Map the data for each row.
     *
     * @param \App\Models\Transaction $transaction
     * @return array
     */
    public function map($transaction): array
    {
        return [
            $transaction->id,
            optional($transaction->user)->name, // Access user name safely
            number_format($transaction->amount, 2),
            ucfirst($transaction->type),
            ucfirst($transaction->status),
            $transaction->created_at->format('d M Y, h:i A'),
        ];
    }

    /**
     * Apply styles to the Excel sheet.
     *
     * @param Worksheet $sheet
     * @return array
     */
    public function styles(Worksheet $sheet)
    {
        return [
            // Styling the heading row
            1 => ['font' => ['bold' => true]],
        ];
    }
}
