<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;

class Referral extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'referrals';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<string>
     */
    protected $fillable = [
        'referrer_id',
        'referred_user_id',
        'earnings',
        'bonus',
        'ib_id',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'earnings' => 'float',
        'bonus' => 'float',
    ];

    /**
     * Relationships
     */

    public function referrer()
    {
        return $this->belongsTo(User::class, 'referrer_id');
    }

    public function referredUser()
    {
        return $this->belongsTo(User::class, 'referred_user_id');
    }

    public function ib()
    {
        return $this->belongsTo(User::class, 'ib_id');
    }

    public function nestedReferrals()
    {
        return $this->hasMany(self::class, 'referrer_id', 'referred_user_id');
    }

    /**
     * Recursive Methods
     */

    public function getAllNestedReferrals(): Collection
    {
        $nestedReferrals = $this->nestedReferrals;

        foreach ($nestedReferrals as $referral) {
            $referral->nestedReferrals = $referral->getAllNestedReferrals();
        }

        return $nestedReferrals;
    }

    public function calculateTotalEarnings(): float
    {
        $totalEarnings = $this->earnings;

        foreach ($this->nestedReferrals as $referral) {
            $totalEarnings += $referral->calculateTotalEarnings();
        }

        return $totalEarnings;
    }

    public function calculateTotalBonuses(): float
    {
        $totalBonuses = $this->bonus;

        foreach ($this->nestedReferrals as $referral) {
            $totalBonuses += $referral->calculateTotalBonuses();
        }

        return $totalBonuses;
    }

    public function getFlatNestedReferrals(): Collection
    {
        $flatReferrals = collect([$this]);

        foreach ($this->nestedReferrals as $referral) {
            $flatReferrals = $flatReferrals->merge($referral->getFlatNestedReferrals());
        }

        return $flatReferrals->filter();
    }

    public function hasActiveReferrals(): bool
    {
        return $this->nestedReferrals->contains(function ($referral) {
            return $referral->referredUser && $referral->referredUser->is_active;
        });
    }

    public function getActiveReferrals(): Collection
    {
        $activeReferrals = $this->nestedReferrals->filter(function ($referral) {
            return $referral->referredUser && $referral->referredUser->is_active;
        });

        foreach ($this->nestedReferrals as $referral) {
            $activeReferrals = $activeReferrals->merge($referral->getActiveReferrals());
        }

        return $activeReferrals;
    }

    /**
     * Attribute Accessors
     */

    public function getFormattedEarningsAttribute(): string
    {
        return number_format($this->earnings, 2);
    }

    public function getFormattedBonusAttribute(): string
    {
        return number_format($this->bonus, 2);
    }

    public function getFormattedCreatedAtAttribute(): string
    {
        return $this->created_at ? $this->created_at->format('d M, Y H:i') : 'N/A';
    }

    /**
     * Scopes
     */

    public function scopeByReferrer($query, int $referrerId)
    {
        return $query->where('referrer_id', $referrerId);
    }

    public function scopeByReferredUser($query, int $referredUserId)
    {
        return $query->where('referred_user_id', $referredUserId);
    }

    public function scopeByMinEarnings($query, float $minEarnings)
    {
        return $query->where('earnings', '>=', $minEarnings);
    }

    /**
     * Utility Methods
     */

    public function updateReferral(array $attributes): bool
    {
        return $this->update($attributes);
    }

    public function isReferredUserActive(): bool
    {
        return $this->referredUser && $this->referredUser->is_active;
    }

    public function getFormattedSummary(): array
    {
        return [
            'referrer' => $this->referrer->name ?? 'Unknown',
            'referred_user' => $this->referredUser->name ?? 'Unknown',
            'earnings' => $this->getFormattedEarningsAttribute(),
            'bonus' => $this->getFormattedBonusAttribute(),
            'status' => $this->isReferredUserActive() ? 'Active' : 'Inactive',
            'created_at' => $this->getFormattedCreatedAtAttribute(),
        ];
    }

    /**
     * Hierarchy Methods
     */

    public function calculateReferralDepth(): int
    {
        return $this->parent ? 1 + $this->parent->calculateReferralDepth() : 1;
    }

    public function hasReferredUser(int $userId): bool
    {
        return $this->referred_user_id === $userId || $this->nestedReferrals->contains('referred_user_id', $userId);
    }
}
