<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;

class IBRelationship extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'ib_relationships';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<string>
     */
    protected $fillable = ['ib_id', 'referred_user_id', 'level'];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'level' => 'integer',
    ];

    /**
     * Define the relationship to the IB (Introducing Broker).
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function ib()
    {
        return $this->belongsTo(User::class, 'ib_id');
    }

    /**
     * Define the relationship to the referred user.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function referredUser()
    {
        return $this->belongsTo(User::class, 'referred_user_id');
    }



    /**
     * Define the relationship for nested referrals under a specific IB.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function nestedReferrals()
    {
        return $this->hasMany(self::class, 'ib_id', 'referred_user_id');
    }

    /**
     * Accessor to get the hierarchy level of the IB relationship.
     *
     * @return int
     */
    public function getHierarchyLevelAttribute(): int
    {
        return $this->level ?? 1;
    }

    /**
     * Retrieve all nested referrals recursively.
     *
     * @return Collection
     */
    public function getAllNestedReferrals(): Collection
    {
        $nestedReferrals = $this->nestedReferrals;

        foreach ($nestedReferrals as $referral) {
            $referral->nestedReferrals = $referral->getAllNestedReferrals();
        }

        return $nestedReferrals;
    }

    /**
     * Count all nested referrals recursively.
     *
     * @return int
     */
    public function countAllNestedReferrals()
    {
        $count = $this->nestedReferrals->count();

        foreach ($this->nestedReferrals as $referral) {
            $count += $referral->countAllNestedReferrals();
        }

        return $count;
    }
    /**
     * Retrieve all IB relationships for a specific user, including nested relationships.
     *
     * @param int $userId
     * @return Collection
     */
    public static function getUserRelationships(int $userId): Collection
    {
        return self::with(['referredUser', 'nestedReferrals'])
            ->where('ib_id', $userId)
            ->get();
    }

    /**
     * Calculate total earnings for an IB across all nested referrals.
     *
     * @return float
     */
    public function calculateTotalEarnings(): float
    {
        $totalEarnings = $this->referredUser->earnings ?? 0;

        foreach ($this->nestedReferrals as $referral) {
            $totalEarnings += $referral->calculateTotalEarnings();
        }

        return $totalEarnings;
    }

    /**
     * Get a flat list of all nested referrals under this IB.
     *
     * @return Collection
     */
    public function getFlatNestedReferrals(): Collection
    {
        $flatReferrals = collect([$this->referredUser]);

        foreach ($this->nestedReferrals as $referral) {
            $flatReferrals = $flatReferrals->merge($referral->getFlatNestedReferrals());
        }

        return $flatReferrals->filter();
    }

    /**
     * Check if the IB has any active referrals.
     *
     * @return bool
     */
    public function hasActiveReferrals(): bool
    {
        return $this->nestedReferrals->contains(function ($referral) {
            return $referral->referredUser && $referral->referredUser->is_active;
        });
    }

    /**
     * Fetch all active referrals recursively.
     *
     * @return Collection
     */
    public function getActiveReferrals(): Collection
    {
        $activeReferrals = $this->nestedReferrals->filter(function ($referral) {
            return $referral->referredUser && $referral->referredUser->is_active;
        });

        foreach ($this->nestedReferrals as $referral) {
            $activeReferrals = $activeReferrals->merge($referral->getActiveReferrals());
        }

        return $activeReferrals;
    }

    /**
     * Fetch referrals by a specific level in the hierarchy.
     *
     * @param int $level
     * @return Collection
     */
    public static function getReferralsByLevel(int $level): Collection
    {
        return self::where('level', $level)->with('referredUser')->get();
    }

    /**
     * Update an existing IB relationship with new attributes.
     *
     * @param array $attributes
     * @return bool
     */
    public function updateRelationship(array $attributes): bool
    {
        return $this->update($attributes);
    }

    /**
     * Get a formatted representation of the relationship's creation date.
     *
     * @return string
     */
    public function getFormattedCreatedAtAttribute(): string
    {
        return $this->created_at->format('d M, Y H:i');
    }

    /**
     * Fetch the IB name with the level info.
     *
     * @return string
     */
    public function getIbNameWithLevelAttribute(): string
    {
        return $this->ib->name . ' (Level ' . $this->level . ')';
    }

    /**
     * Static method to find and update a specific IB relationship.
     *
     * @param int $id
     * @param array $attributes
     * @return bool
     */
    public static function findAndUpdate(int $id, array $attributes): bool
    {
        $relationship = self::find($id);
        if ($relationship) {
            return $relationship->update($attributes);
        }
        return false;
    }

    /**
     * Scope to filter relationships by active status.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param bool $isActive
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeIsActive($query, bool $isActive)
    {
        return $query->whereHas('referredUser', function ($query) use ($isActive) {
            $query->where('is_active', $isActive);
        });
    }


    /**
     * Scope to filter IB relationships by a specific IB.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param int $ibId
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeByIbId($query, int $ibId)
    {
        return $query->where('ib_id', $ibId);
    }

    /**
     * Utility method to format IB relationship details for display.
     *
     * @return array
     */
    public function formatRelationshipDetails(): array
    {
        return [
            'ib_name' => $this->ib->name ?? 'Unknown',
            'referred_user_name' => $this->referredUser->name ?? 'Unknown',
            'level' => $this->level,
            'active' => $this->referredUser && $this->referredUser->is_active ? 'Active' : 'Inactive',
            'created_at' => $this->getFormattedCreatedAtAttribute(),
        ];
    }
}
