<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Collection;

class IBHierarchy extends Model
{
    /**
     * Table associated with the model.
     *
     * @var string
     */
    protected $table = 'ib_hierarchy';

    /**
     * Attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'ib_id',
        'parent_id',
        'level',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'level' => 'integer',
    ];

    /**
     * Define the relationship to the IB user.
     *
     * @return BelongsTo
     */
    public function ib(): BelongsTo
    {
        return $this->belongsTo(User::class, 'ib_id');
    }

    /**
     * Define the relationship to the parent IB user.
     *
     * @return BelongsTo
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo(User::class, 'parent_id');
    }

    /**
     * Fetch the children of the IB.
     *
     * @return HasMany
     */
    public function children(): HasMany
    {
        return $this->hasMany(IBHierarchy::class, 'parent_id');
    }

    /**
     * Retrieve all nested children recursively.
     *
     * @return Collection
     */
    public function getAllNestedChildren(): Collection
    {
        $children = $this->children;

        foreach ($children as $child) {
            $child->children = $child->getAllNestedChildren();
        }

        return $children;
    }

    /**
     * Retrieve the full flat list of all nested children.
     *
     * @return Collection
     */
    public function getFlatChildren(): Collection
    {
        $flatChildren = collect([$this->ib]);
        foreach ($this->children as $child) {
            $flatChildren = $flatChildren->merge($child->getFlatChildren());
        }

        return $flatChildren->filter();
    }

    /**
     * Check if an IB has any children.
     *
     * @return bool
     */
    public function hasChildren(): bool
    {
        return $this->children()->exists();
    }

    /**
     * Count the total number of all nested children.
     *
     * @return int
     */
    public function countAllNestedChildren(): int
    {
        $count = $this->children->count();
        foreach ($this->children as $child) {
            $count += $child->countAllNestedChildren();
        }

        return $count;
    }

    /**
     * Retrieve the level of the IB in the hierarchy.
     *
     * @return int
     */
    public function getLevelAttribute(): int
    {
        return $this->attributes['level'] ?? 1;
    }

    /**
     * Get formatted creation date for display.
     *
     * @return string
     */
    public function getFormattedCreatedAtAttribute(): string
    {
        return $this->created_at ? $this->created_at->format('d M, Y H:i') : 'N/A';
    }

    /**
     * Define the relationship to the parent IB hierarchy record.
     *
     * @return BelongsTo
     */
    public function parentHierarchy(): BelongsTo
    {
        return $this->belongsTo(IBHierarchy::class, 'parent_id');
    }

    /**
     * Fetch the root IB for the hierarchy (top-most ancestor).
     *
     * @return IBHierarchy
     */
    public function getRootIb(): IBHierarchy
    {
        return $this->parentHierarchy ? $this->parentHierarchy->getRootIb() : $this;
    }

    /**
     * Calculate the depth of this IB in the hierarchy.
     *
     * @return int
     */
    public function calculateDepth(): int
    {
        return $this->parentHierarchy ? 1 + $this->parentHierarchy->calculateDepth() : 1;
    }

    /**
     * Scopes
     */

    /**
     * Scope to filter IB relationships by a specific parent.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param int $parentId
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeByParent($query, int $parentId)
    {
        return $query->where('parent_id', $parentId);
    }

    /**
     * Scope to filter IB relationships by level.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param int $level
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeByLevel($query, int $level)
    {
        return $query->where('level', $level);
    }

    /**
     * Utility Methods
     */

    /**
     * Update an existing IB hierarchy with new attributes.
     *
     * @param array $attributes
     * @return bool
     */
    public function updateHierarchy(array $attributes): bool
    {
        return $this->update($attributes);
    }

    /**
     * Fetch all active children recursively.
     *
     * @return Collection
     */
    public function getActiveChildren(): Collection
    {
        $activeChildren = $this->children->filter(function ($child) {
            return $child->ib && $child->ib->is_active;
        });

        foreach ($this->children as $child) {
            $activeChildren = $activeChildren->merge($child->getActiveChildren());
        }

        return $activeChildren;
    }

    /**
     * Retrieve a formatted list of IB children with levels.
     *
     * @return Collection
     */
    public function getFormattedHierarchy(): Collection
    {
        $children = $this->getAllNestedChildren();
        return $children->map(function ($child) {
            return [
                'ib_name' => $child->ib->name ?? 'Unknown',
                'level' => $child->level,
                'active' => $child->ib->is_active ? 'Active' : 'Inactive',
                'created_at' => $child->getFormattedCreatedAtAttribute(),
            ];
        });
    }

    /**
     * Check if this IB is a root IB (has no parent).
     *
     * @return bool
     */
    public function isRootIb(): bool
    {
        return $this->parent_id === null;
    }
}
