<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Transaction;
use App\Models\Wallet;
use App\Models\ActivityLog;
use App\Models\PaymentGatewaySetting;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Services\PaymentGatewayService;

class WalletController extends Controller
{
    protected $paymentGatewayService;

    public function __construct(PaymentGatewayService $paymentGatewayService)
    {
        $this->paymentGatewayService = $paymentGatewayService;
    }

    /**
     * Display the wallet page with balance, transactions, and analytics.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $user = Auth::user();
        $wallet = Wallet::firstOrCreate(['user_id' => $user->id]);
        $transactions = Transaction::where('user_id', $user->id)->latest()->get();
        $recentTransaction = $transactions->first();
        $activeGateways = PaymentGatewaySetting::where('is_active', 1)->get();

        return view('user.wallet', [
            'wallet' => $wallet,
            'transactions' => $transactions,
            'recentActivities' => $transactions->take(5),
            'recent_transaction' => $recentTransaction ? $recentTransaction->type . " of $" . number_format($recentTransaction->amount, 2) : null,
            'activeGateways' => $activeGateways,
        ]);
    }

    /**
     * Display the deposit page.
     *
     * @return \Illuminate\View\View
     */
    public function deposit()
    {
        $activeGateways = PaymentGatewaySetting::where('is_active', 1)->get();
        return view('user.deposit', compact('activeGateways'));
    }


    /**
     * Initiate a deposit for the user.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse
     */
    public function initiateDeposit(Request $request)
    {
        // Validate request
        $request->validate([
            'amount' => 'required|numeric|min:1',
            'payment_method' => 'required|string',
        ]);

        $amount = $request->input('amount');
        $paymentMethod = $request->input('payment_method');

        // Perform logic to initiate the deposit (e.g., redirect to payment gateway)
        // Here, you can interact with your payment service or gateway
        $response = $this->processPayment($amount, $paymentMethod);

        // Example response handling
        if ($response['status'] === 'success') {
            return redirect()->route('wallet.index')->with('success', 'Deposit initiated successfully.');
        }

        return redirect()->back()->withErrors('Failed to initiate deposit. Please try again.');
    }

    /**
     * Mock payment processing method (replace with real gateway logic).
     *
     * @param float $amount
     * @param string $paymentMethod
     * @return array
     */
    private function processPayment($amount, $paymentMethod)
    {
        // Example response from a payment gateway
        return [
            'status' => 'success',
            'transaction_id' => 'TXN12345678',
        ];
    }



    /**
     * Process deposit request - Step 1: Initiate Payment.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function processDeposit(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:10',
            'payment_gateway' => 'required|string|exists:payment_gateway_settings,gateway_name',
        ]);

        $user = Auth::user();
        $amount = $request->amount;
        $paymentGateway = $request->payment_gateway;

        $gateway = PaymentGatewaySetting::where('gateway_name', $paymentGateway)
            ->where('is_active', 1)
            ->first();

        if (!$gateway) {
            return back()->with('error', 'Selected payment gateway is not available.');
        }

        // Create a pending transaction
        $transaction = Transaction::create([
            'user_id' => $user->id,
            'type' => 'deposit',
            'amount' => $amount,
            'status' => 'pending',
            'payment_gateway' => $paymentGateway,
        ]);

        // Check if it's a manual payment gateway
        if ($gateway->gateway_type === 'manual') {
            return redirect()->route('wallet.manualPaymentDetails', ['transactionId' => $transaction->id]);
        }

        try {
            // Handle automated payment gateway logic
            $paymentUrl = $this->paymentGatewayService->initiatePayment($transaction, $gateway);
            return redirect($paymentUrl);
        } catch (\Exception $e) {
            return back()->with('error', 'Error initiating payment: ' . $e->getMessage());
        }
    }

    /**
     * Display manual payment details.
     *
     * @param int $transactionId
     * @return \Illuminate\View\View
     */
    public function manualPaymentDetails($transactionId)
    {
        $transaction = Transaction::findOrFail($transactionId);
        $gateway = PaymentGatewaySetting::where('gateway_name', $transaction->payment_gateway)->first();

        return view('user.manual_payment_details', [
            'transaction' => $transaction,
            'gateway' => $gateway,
        ]);
    }

    /**
     * Handle payment callback - Step 3: Payment Confirmation.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $transactionId
     * @return \Illuminate\Http\RedirectResponse
     */
    public function handlePaymentCallback(Request $request, $transactionId)
    {
        $transaction = Transaction::findOrFail($transactionId);

        if ($transaction->status !== 'completed') {
            DB::transaction(function () use ($transaction) {
                // Update wallet balance
                $wallet = Wallet::firstOrCreate(['user_id' => $transaction->user_id]);
                $wallet->balance += $transaction->amount;
                $wallet->save();

                // Update transaction status
                $transaction->status = 'completed';
                $transaction->save();

                // Log activity
                ActivityLog::create([
                    'user_id' => $transaction->user_id,
                    'activity_type' => 'deposit',
                    'details' => "Deposit of amount: " . $transaction->amount,
                    'ip_address' => request()->ip(),
                ]);
            });
        }

        return redirect()->route('wallet.index')->with('success', 'Deposit successful.');
    }


    /**
     * Handle withdrawal requests - Step 1: Initiate Withdrawal.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function initiateWithdraw(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:10',
            'payment_method' => 'required|string',
        ]);

        $user = Auth::user();
        $amount = $request->amount;
        $paymentMethod = $request->payment_method;

        $wallet = Wallet::where('user_id', $user->id)->first();

        if (!$wallet || $wallet->balance < $amount) {
            return back()->with('error', 'Insufficient wallet balance.');
        }

        $transaction = Transaction::create([
            'user_id' => $user->id,
            'type' => 'withdrawal',
            'amount' => $amount,
            'status' => 'pending',
            'payment_gateway' => $paymentMethod,
        ]);

        ActivityLog::create([
            'user_id' => $user->id,
            'activity_type' => 'withdrawal_request',
            'details' => "Withdrawal request for amount: $amount",
            'ip_address' => $request->ip(),
        ]);

        return back()->with('success', 'Withdrawal request submitted successfully and awaiting approval.');
    }

    /**
     * Admin approves withdrawal - Step 2: Admin Processing.
     *
     * @param int $transactionId
     * @return \Illuminate\Http\RedirectResponse
     */
    public function approveWithdraw($transactionId)
    {
        $transaction = Transaction::findOrFail($transactionId);

        if ($transaction->status !== 'pending') {
            return back()->with('error', 'Withdrawal request is not in pending state.');
        }

        DB::transaction(function () use ($transaction) {
            $transaction->status = 'approved';
            $transaction->save();

            $wallet = Wallet::where('user_id', $transaction->user_id)->first();
            $wallet->balance -= $transaction->amount;
            $wallet->save();

            ActivityLog::create([
                'user_id' => $transaction->user_id,
                'activity_type' => 'withdrawal_approved',
                'details' => "Approved withdrawal for amount: " . $transaction->amount,
                'ip_address' => request()->ip(),
            ]);
        });

        return back()->with('success', 'Withdrawal approved successfully.');
    }

    /**
     * Admin rejects withdrawal - Step 3: Admin Processing.
     *
     * @param int $transactionId
     * @return \Illuminate\Http\RedirectResponse
     */
    public function rejectWithdraw($transactionId)
    {
        $transaction = Transaction::findOrFail($transactionId);

        if ($transaction->status !== 'pending') {
            return back()->with('error', 'Withdrawal request is not in pending state.');
        }

        $transaction->status = 'rejected';
        $transaction->save();

        ActivityLog::create([
            'user_id' => $transaction->user_id,
            'activity_type' => 'withdrawal_rejected',
            'details' => "Rejected withdrawal for amount: " . $transaction->amount,
            'ip_address' => request()->ip(),
        ]);

        return back()->with('success', 'Withdrawal rejected successfully.');
    }

    /**
     * Handle PayPal payment initiation.
     *
     * @param int $transactionId
     * @return \Illuminate\Http\RedirectResponse
     */
    public function initiatePaypalPayment($transactionId)
    {
        $transaction = Transaction::findOrFail($transactionId);

        // PayPal payment initiation logic here
        return redirect()->to('https://www.paypal.com/cgi-bin/webscr?...'); // Replace with real PayPal initiation link
    }
}
