<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use App\Models\Wallet;
use App\Models\Trade;
use App\Models\Referral;
use App\Models\ActivityLog;
use App\Models\IBRelationship;
use App\Models\Transaction;
use App\Services\MT5ManagerService;
use App\Mail\UserNotificationMail;
use Illuminate\Support\Facades\Mail;

class UserDashboardController extends Controller
{
    protected $mt5Service;

    /**
     * Inject MT5ManagerService in the constructor.
     */
    public function __construct(MT5ManagerService $mt5Service)
    {
        $this->mt5Service = $mt5Service;
    }

    /**
     * Display the user dashboard.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        Gate::authorize('view-user-dashboard');

        $user = Auth::user();

        // Dashboard Data
        $walletBalance = Wallet::where('user_id', $user->id)->sum('balance');
        $openTrades = Trade::where('user_id', $user->id)->where('status', 'open')->count();
        $referralEarnings = Referral::where('referrer_id', $user->id)->sum('earnings');
        $recentTrades = Trade::where('user_id', $user->id)->latest()->take(5)->get();

        // Calculate total referrals recursively
        $totalReferrals = $this->calculateTotalReferrals($user->id);

        // Monthly Profit/Loss Data
        $monthlyProfitLoss = Trade::where('user_id', $user->id)
            ->selectRaw('DATE_FORMAT(created_at, "%Y-%m") as month, COALESCE(SUM(profit_loss), 0) as total_profit_loss')
            ->groupBy('month')
            ->orderBy('month', 'asc')
            ->pluck('total_profit_loss', 'month')
            ->toArray();
        $monthlyProfitLoss = !empty($monthlyProfitLoss) ? $monthlyProfitLoss : ['No Data' => 0];

        // Funds Transfer Data
        $deposits = Transaction::where('user_id', $user->id)->where('type', 'deposit')->sum('amount');
        $withdrawals = Transaction::where('user_id', $user->id)->where('type', 'withdraw')->sum('amount');
        $fundsTransferData = [$deposits, $withdrawals];


        // Deposit vs Withdrawal Data
        $deposits = Transaction::where('user_id', $user->id)->where('type', 'deposit')->sum('amount');
        $withdrawals = Transaction::where('user_id', $user->id)->where('type', 'withdraw')->sum('amount');
        $depositWithdrawData = [$deposits, $withdrawals];

        // Top Traded Assets Data
        $topAssets = Trade::where('user_id', $user->id)
            ->selectRaw('symbol as asset, COUNT(symbol) as count')
            ->groupBy('symbol')
            ->orderByDesc('count')
            ->take(5)
            ->pluck('count', 'asset')
            ->toArray();
        $topAssets = !empty($topAssets) ? $topAssets : ['No Data' => 0];

        return view('user.dashboard', [
            'walletBalance' => $walletBalance,
            'openTrades' => $openTrades,
            'referralEarnings' => $referralEarnings,
            'recentTrades' => $recentTrades,
            'monthlyProfitLossLabels' => array_keys($monthlyProfitLoss),
            'monthlyProfitLossData' => array_values($monthlyProfitLoss),
            'topAssetsLabels' => array_keys($topAssets),
            'topAssetsData' => array_values($topAssets),
            'totalReferrals' => $totalReferrals,
            'depositWithdrawData' => $depositWithdrawData,
            'fundsTransferData' => $fundsTransferData, // Pass this variable to the view
        ]);
    }

    /**
     * Calculate total referrals recursively, including nested levels.
     *
     * @param int $userId
     * @return int
     */
    private function calculateTotalReferrals($userId)
    {
        $directReferrals = IBRelationship::where('ib_id', $userId)->get();

        $count = $directReferrals->count();

        foreach ($directReferrals as $referral) {
            $count += $this->calculateTotalReferrals($referral->referred_user_id);
        }

        return $count;
    }

    /**
     * Display user activity logs.
     *
     * @return \Illuminate\View\View
     */
    public function activityLogs(Request $request)
    {
        $search = $request->get('search');

        $logs = ActivityLog::where('user_id', Auth::id())
            ->when($search, function ($query, $search) {
                $query->where('action', 'LIKE', '%' . $search . '%');
            })
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('user.activity-logs', compact('logs'));
    }

    /**
     * Display user's trades with optional filters.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\View\View
     */
    public function trades(Request $request)
    {
        Gate::authorize('view-trades');

        $user = Auth::user();
        $query = Trade::where('user_id', $user->id);

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('asset')) {
            $query->where('asset', 'like', '%' . $request->asset . '%');
        }

        $trades = $query->latest()->paginate(10);

        return view('user.trades', compact('trades'));
    }

    /**
     * Show the transfer funds page.
     *
     * @return \Illuminate\View\View
     */
    public function showTransferFunds()
    {
        Gate::authorize('transfer-funds');

        $walletBalance = Auth::user()->wallet->balance;

        return view('user.transferFunds', compact('walletBalance'));
    }

    /**
     * Show the page to create a live account.
     *
     * @return \Illuminate\View\View
     */
    public function showCreateLiveAccount()
    {
        return view('user.mt5_create_live');
    }

    /**
     * Create a live MT5 account for the user.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function createLiveAccount(Request $request)
    {
        $request->validate([
            'leverage' => 'required|integer|min:50|max:500',
            'group' => 'required|string|max:50',
        ]);

        $user = Auth::user();

        try {
            $response = $this->mt5Service->createAccount([
                'name' => $user->name,
                'group' => $request->group,
                'leverage' => $request->leverage,
                'main_password' => 'UserMainPassword123',
                'investor_password' => 'InvestorPass123',
                'phone_password' => 'PhonePass123',
                'account_type' => 'live',
            ]);

            if ($response['status'] === 'success') {
                return redirect()->back()->with('success', 'Live MT5 account created successfully!');
            }

            return redirect()->back()->with('error', $response['message'] ?? 'Failed to create live account.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }

    /**
     * Create a demo MT5 account for the user.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function createDemoAccount(Request $request)
    {
        $request->validate([
            'leverage' => 'required|integer|min:50|max:500',
            'group' => 'required|string|max:50',
        ]);

        $user = Auth::user();

        try {
            $response = $this->mt5Service->createAccount([
                'name' => $user->name,
                'group' => $request->group,
                'leverage' => $request->leverage,
                'main_password' => 'UserMainPassword123',
                'investor_password' => 'InvestorPass123',
                'phone_password' => 'PhonePass123',
                'account_type' => 'demo',
            ]);

            if ($response['status'] === 'success') {
                return redirect()->back()->with('success', 'Demo MT5 account created successfully!');
            }

            return redirect()->back()->with('error', $response['message'] ?? 'Failed to create demo account.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }




    /**
     * Show the user's MT5 trades.
     *
     * @return \Illuminate\View\View
     */
    public function showTrades()
    {
        $accountId = Auth::user()->mt5_account_id;

        $response = $this->mt5Service->getTrades($accountId);

        $trades = $response['status'] === 'success' ? $response['data'] : [];

        return view('user.mt5_trades', compact('trades'));
    }


    //sendmail
    public function sendWelcomeEmail($user)
    {
        $content = [
            'subject' => 'Welcome to Our Platform',
            'title' => 'Welcome Aboard!',
            'message' => 'Thank you for signing up, ' . $user->name . '!'
        ];

        Mail::to($user->email)->send(new UserNotificationMail($content));
    }


    public function transferFunds(Request $request)
    {
        $request->validate([
            'login' => 'required|string',
            'amount' => 'required|numeric|min:1',
            'action' => 'required|in:deposit,withdraw',
        ]);

        try {
            // Calling the transferFunds method in MT5ManagerService
            $response = $this->mt5Service->transferFunds(
                $request->login,
                $request->amount,
                $request->action === 'deposit' ? 'Deposit' : 'Withdrawal',
                $request->action
            );

            if ($response['status'] === 'success') {
                return redirect()->back()->with('success', ucfirst($request->action) . ' completed successfully!');
            }

            return redirect()->back()->with('error', $response['message'] ?? 'Failed to complete transaction.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }
}
