<?php

namespace App\Http\Controllers;

use App\Models\SupportTicket;
use App\Models\SupportTicketReply;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class SupportTicketController extends Controller
{
    /**
     * Display a listing of the user's support tickets.
     *
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $query = SupportTicket::where('user_id', Auth::id());

        // Search filter
        if ($request->has('search') && $request->search != '') {
            $query->where('subject', 'like', '%' . $request->search . '%');
        }

        // Get paginated results
        $tickets = $query->orderBy('created_at', 'desc')->paginate(10);

        return view('user.support.index', compact('tickets'));
    }

    /**
     * Show the form for creating a new support ticket.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        return view('user.support.create');
    }

    /**
     * Store a newly created support ticket in the database.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $request->validate([
            'subject' => 'required|string|max:255',
            'description' => 'required|string|min:10',
            'attachments.*' => 'file|mimes:jpg,jpeg,png,pdf,docx|max:2048',
        ]);

        $ticket = SupportTicket::create([
            'user_id' => Auth::id(),
            'subject' => $request->subject,
            'description' => $request->description,
            'status' => 'open', // Default status for new tickets
        ]);

        // Handle file uploads
        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $filePath = $file->store('support_tickets', 'public');
                $ticket->attachments()->create([
                    'file_path' => $filePath,
                    'file_name' => $file->getClientOriginalName(),
                ]);
            }
        }

        // Log activity
        Log::info('New support ticket created.', [
            'user_id' => Auth::id(),
            'ticket_id' => $ticket->id,
        ]);

        return redirect()->route('user.support.index')->with('success', 'Support ticket created successfully.');
    }

    /**
     * Display the details of a specific support ticket and its replies.
     *
     * @param int $id
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        $ticket = SupportTicket::with(['replies.user'])->where('user_id', Auth::id())->findOrFail($id);

        return view('user.support.show', compact('ticket'));
    }

    /**
     * Store a reply to a specific support ticket.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function reply(Request $request, $id)
    {
        $request->validate([
            'message' => 'required|string|max:1000',
        ]);

        $ticket = SupportTicket::where('user_id', Auth::id())->findOrFail($id);

        // Check if ticket is still open
        if (in_array($ticket->status, ['completed', 'closed'])) {
            return back()->with('error', 'You cannot reply to a closed or completed ticket.');
        }

        SupportTicketReply::create([
            'ticket_id' => $ticket->id,
            'user_id' => Auth::id(),
            'message' => $request->message,
        ]);

        // Log activity
        Log::info('User replied to a support ticket.', [
            'user_id' => Auth::id(),
            'ticket_id' => $ticket->id,
        ]);

        return back()->with('success', 'Your reply has been sent successfully.');
    }
}
