<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\IBRelationship;
use App\Models\Referral;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class IBDashboardController extends Controller
{
    /**
     * Display the IB Dashboard with various metrics.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $user = Auth::user();

        // Direct referrals count
        $directReferralsCount = Referral::where('referrer_id', $user->id)->count();

        // Total referrals count (includes nested referrals)
        $totalReferralsCount = $this->calculateTotalReferrals($user->id);

        // Total earnings
        $totalEarnings = Referral::where('referrer_id', $user->id)->sum('earnings');

        // Fetch direct referrals for display
        $referrals = Referral::where('referrer_id', $user->id)
            ->with('referredUser')
            ->get();

        return view('user.ib-dashboard', compact(
            'user',
            'directReferralsCount',
            'totalReferralsCount',
            'totalEarnings',
            'referrals'
        ));
    }

    /**
     * Calculate total referrals, including nested referrals.
     *
     * @param int $ibId
     * @return int
     */
    private function calculateTotalReferrals($ibId)
    {
        $directReferrals = Referral::where('referrer_id', $ibId)->with('referredUser.referrals')->get();
        $totalCount = $directReferrals->count();

        foreach ($directReferrals as $referral) {
            if ($referral->referredUser) {
                $nestedReferrals = $referral->referredUser->referrals ?? collect();
                $totalCount += $nestedReferrals->count();
            }
        }

        return $totalCount;
    }

    /**
     * Display referral earnings for the IB.
     *
     * @return \Illuminate\View\View
     */
    public function referralEarnings()
    {
        $user = Auth::user();

        // Calculate total earnings
        $totalEarnings = Referral::where('referrer_id', $user->id)->sum('earnings');

        return view('user.referral-earnings', compact('user', 'totalEarnings'));
    }

    /**
     * Get referral growth data for the past 6 months.
     *
     * @param int $ibId
     * @return array
     */
    private function getReferralGrowthData($ibId)
    {
        $growthData = [];
        $currentMonth = Carbon::now();

        for ($i = 5; $i >= 0; $i--) {
            $month = $currentMonth->copy()->subMonths($i);
            $referralCount = Referral::where('referrer_id', $ibId)
                ->whereMonth('created_at', $month->month)
                ->whereYear('created_at', $month->year)
                ->count();

            $growthData[] = [
                'month' => $month->format('F'),
                'count' => $referralCount,
            ];
        }

        return $growthData;
    }

    /**
     * Display active referrals for the IB.
     *
     * @return \Illuminate\View\View
     */
    public function activeReferrals()
    {
        $user = Auth::user();

        // Fetch active referrals
        $activeReferrals = Referral::where('referrer_id', $user->id)
            ->whereHas('referredUser', function ($query) {
                $query->where('is_active', true);
            })
            ->with('referredUser')
            ->get();

        return view('user.active-referrals', compact('user', 'activeReferrals'));
    }

    /**
     * Display earnings for the IB.
     *
     * @return \Illuminate\View\View
     */
    public function earnings()
    {
        $user = Auth::user();

        // Direct and indirect earnings
        $directEarnings = Referral::where('referrer_id', $user->id)->sum('earnings');
        $totalEarnings = $directEarnings;

        return view('user.earnings', compact('user', 'directEarnings', 'totalEarnings'));
    }
}
