<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\IBRelationship;
use App\Models\IBHierarchy;
use App\Models\Referral;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use App\Models\IBCommissionLevel;

class IBController extends Controller
{
    /**
     * Display the IB management dashboard with filtering and pagination.
     */
    public function index(Request $request)
    {
        $query = User::where('is_ib', true);

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                    ->orWhere('email', 'like', '%' . $request->search . '%');
            });
        }

        if ($request->filled('status')) {
            $isActive = $request->status === 'active';
            $query->where('is_active', $isActive);
        }

        $ibs = $query->with(['referredUsers', 'parentIB'])->paginate(10);

        return view('admin.ib-management', compact('ibs'));
    }

    /**
     * Show the IB tree for a specific IB.
     */
    public function showTree($id)
    {
        $ib = User::findOrFail($id);

        if (!$ib->is_ib) {
            return redirect()->route('admin.ib.management')->with('error', 'User is not an IB.');
        }

        $tree = $this->buildTree($ib->id);

        return view('admin.ib-tree', compact('ib', 'tree'));
    }

    /**
     * Assign a user as an IB under another IB.
     */
    public function assignIB(Request $request)
    {
        $request->validate([
            'ib_id' => 'required|exists:users,id',
            'user_id' => 'required|exists:users,id',
            'trade_amount' => 'required|numeric|min:0', // Assuming trade amount is passed in request
        ]);

        // Retrieve the IB user
        $ib = User::findOrFail($request->ib_id);

        // Retrieve commission level for this IB's level
        $commissionLevel = IBCommissionLevel::where('level', $ib->level)->first();
        $commissionPercentage = $commissionLevel ? $commissionLevel->commission_percentage : 0;

        // Calculate earnings
        $earnings = $request->trade_amount * ($commissionPercentage / 100);

        // Create or update referral relationship
        IBRelationship::updateOrCreate(
            ['referred_user_id' => $request->user_id],
            [
                'ib_id' => $request->ib_id,
                'level' => $this->calculateLevel($request->ib_id),
            ]
        );

        // Update earnings for the IB in the referral record
        Referral::updateOrCreate(
            ['referrer_id' => $ib->id, 'referred_user_id' => $request->user_id],
            [
                'earnings' => $earnings,
                'bonus' => 0, // Adjust bonus logic if needed
            ]
        );

        User::where('id', $request->user_id)->update(['is_ib' => true]);

        return redirect()->route('admin.ib.management')->with('success', 'User successfully assigned as an IB and earnings calculated.');
    }

    /**
     * Build the IB tree recursively.
     */
    private function buildTree($ibId, $level = 1)
    {
        $children = IBRelationship::where('ib_id', $ibId)->with('referredUser')->get();

        $tree = [];
        foreach ($children as $child) {
            if ($child->referredUser) {
                $tree[] = [
                    'user' => $child->referredUser,
                    'level' => $level,
                    'children' => $this->buildTree($child->referred_user_id, $level + 1),
                ];
            }
        }

        return $tree;
    }

    /**
     * Calculate the level of a user in the IB hierarchy.
     */
    private function calculateLevel($ibId)
    {
        $parent = IBHierarchy::where('ib_id', $ibId)->first();
        return $parent ? $parent->level + 1 : 1;
    }

    /**
     * Create a new IB.
     */
    public function create()
    {
        $ibOptions = User::where('is_ib', true)
            ->where('is_active', true)
            ->get(); // Get list of IBs to select as potential uplines

        return view('admin.ib.create', compact('ibOptions'));
    }

    /**
     * Store a newly created IB in the database.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8|confirmed',
            'upline_ib' => 'nullable|exists:users,id',
        ]);

        // Create the IB user
        $ib = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'is_ib' => true,
            'is_active' => true,
        ]);

        // Create IBHierarchy and IBRelationship if upline is selected
        if ($request->upline_ib) {
            IBHierarchy::create([
                'ib_id' => $ib->id,
                'parent_id' => $request->upline_ib,
                'level' => $this->calculateLevel($request->upline_ib),
            ]);

            IBRelationship::create([
                'ib_id' => $request->upline_ib,
                'referred_user_id' => $ib->id,
                'level' => $this->calculateLevel($request->upline_ib),
            ]);
        }

        return redirect()->route('admin.ib.management')->with('success', 'IB created successfully.');
    }

    /**
     * Edit a specific IB.
     */
    public function edit($id)
    {
        $ib = User::where('is_ib', true)->findOrFail($id);

        // Fetch all active IBs except the current one (to avoid selecting itself as upline)
        $ibOptions = User::where('is_ib', true)
            ->where('is_active', true)
            ->where('id', '!=', $id)
            ->get();

        $levels = [1, 2, 3, 4, 5]; // Example levels, you can fetch levels dynamically if needed

        return view('admin.ib.edit', compact('ib', 'ibOptions', 'levels'));
    }

    /**
     * Update a specific IB.
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'is_active' => 'required|boolean',
            'upline_ib' => 'nullable|exists:users,id',
        ]);

        $ib = User::findOrFail($id);

        if (!$ib->is_ib) {
            return redirect()->route('admin.ib.management')->with('error', 'User is not an IB.');
        }

        // Update user details
        $ib->update($request->only(['name', 'email', 'is_active']));

        // Update IB hierarchy and relationship for upline changes
        if ($request->upline_ib) {
            IBHierarchy::updateOrCreate(
                ['ib_id' => $ib->id],
                [
                    'parent_id' => $request->upline_ib,
                    'level' => $this->calculateLevel($request->upline_ib),
                ]
            );

            IBRelationship::updateOrCreate(
                ['referred_user_id' => $ib->id],
                [
                    'ib_id' => $request->upline_ib,
                    'level' => $this->calculateLevel($request->upline_ib),
                ]
            );
        }

        return redirect()->route('admin.ib.management')->with('success', 'IB updated successfully.');
    }

    /**
     * Update the status of an IB (active/inactive).
     */
    public function updateStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:active,inactive',
        ]);

        $user = User::findOrFail($id);

        if (!$user->is_ib) {
            return redirect()->back()->with('error', 'User is not an IB.');
        }

        $user->is_active = $request->status === 'active';
        $user->save();

        return redirect()->route('admin.ib.management')->with('success', 'IB status updated successfully.');
    }

    /**
     * Display IB earnings.
     */
    public function myEarnings()
    {
        $earnings = Referral::where('ib_id', Auth::id())->paginate(10);
        return view('ib.earnings.index', compact('earnings'));
    }

    /**
     * Delete an IB.
     */
    public function destroy($id)
    {
        $ib = User::where('is_ib', true)->findOrFail($id);

        if (!$ib) {
            return redirect()->route('admin.ib.management')->with('error', 'User not found.');
        }

        // Remove all relationships
        IBHierarchy::where('ib_id', $ib->id)->delete();
        IBRelationship::where('ib_id', $ib->id)->orWhere('referred_user_id', $ib->id)->delete();

        $ib->delete();

        return redirect()->route('admin.ib.management')->with('success', 'IB deleted successfully.');
    }
}
