<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Transaction;
use App\Models\ActivityLog;
use Illuminate\Support\Facades\DB;

class AdminWithdrawController extends Controller
{
    /**
     * Display a list of pending withdrawal requests, along with approved and rejected statistics.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $withdrawals = Transaction::where('type', 'withdrawal')
            ->where('status', 'pending')
            ->latest()
            ->get();

        // Calculate counts for approved and rejected withdrawals
        $approvedWithdrawalsCount = Transaction::where('type', 'withdrawal')
            ->where('status', 'approved')
            ->count();

        $rejectedWithdrawalsCount = Transaction::where('type', 'withdrawal')
            ->where('status', 'rejected')
            ->count();

        return view('admin.withdrawals.index', compact('withdrawals', 'approvedWithdrawalsCount', 'rejectedWithdrawalsCount'));
    }

    /**
     * Approve a pending withdrawal request.
     *
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function approve($id)
    {
        $transaction = Transaction::findOrFail($id);

        if ($transaction->status !== 'pending') {
            return back()->with('error', 'Withdrawal request is not in pending state.');
        }

        DB::transaction(function () use ($transaction) {
            // Mark transaction as approved
            $transaction->status = 'approved';
            $transaction->save();

            // Update wallet balance to reflect the withdrawal
            $wallet = $transaction->user->wallet;

            if (!$wallet || $wallet->balance < $transaction->amount) {
                throw new \Exception('Insufficient wallet balance to approve withdrawal.');
            }

            $wallet->balance -= $transaction->amount;
            $wallet->save();

            // Mark the transaction as completed
            $transaction->status = 'completed';
            $transaction->save();

            // Log withdrawal approval activity
            ActivityLog::create([
                'user_id' => $transaction->user_id,
                'activity_type' => 'withdrawal_approved',
                'details' => "Approved withdrawal for amount: " . $transaction->amount,
                'ip_address' => request()->ip(),
            ]);
        });

        return back()->with('success', 'Withdrawal approved successfully.');
    }

    /**
     * Reject a pending withdrawal request.
     *
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function reject($id)
    {
        $transaction = Transaction::findOrFail($id);

        if ($transaction->status !== 'pending') {
            return back()->with('error', 'Withdrawal request is not in pending state.');
        }

        // Mark transaction as rejected
        $transaction->status = 'rejected';
        $transaction->save();

        // Log withdrawal rejection activity
        ActivityLog::create([
            'user_id' => $transaction->user_id,
            'activity_type' => 'withdrawal_rejected',
            'details' => "Rejected withdrawal for amount: " . $transaction->amount,
            'ip_address' => request()->ip(),
        ]);

        return back()->with('success', 'Withdrawal rejected successfully.');
    }
}
