<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\KycDocument;
use App\Notifications\KycStatusUpdated;
use Symfony\Component\HttpFoundation\StreamedResponse;
use League\Csv\Writer;
use SplTempFileObject;

class AdminKycController extends Controller
{
    /**
     * Display a list of KYC submissions with optional filtering by status.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $query = KycDocument::with('user');

        // Filter by status if provided
        if ($request->has('status') && in_array($request->status, ['pending', 'approved', 'rejected'])) {
            $query->where('status', $request->status);
        }

        $kycDocuments = $query->latest()->paginate(10);

        return view('admin.kyc.index', compact('kycDocuments'));
    }

    /**
     * Show details of a specific KYC submission.
     *
     * @param int $id
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        $kyc = KycDocument::with('user')->findOrFail($id);
        return view('admin.kyc.show', compact('kyc'));
    }

    /**
     * Approve a KYC submission.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function approve(Request $request, $id)
    {
        $kyc = KycDocument::findOrFail($id);
        $kyc->update(['status' => 'approved']);

        // Notify the user
        $kyc->user->notify(new KycStatusUpdated('approved'));

        return redirect()->route('admin.kyc.index')->with('success', 'KYC approved successfully!');
    }

    /**
     * Reject a KYC submission with a reason.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function reject(Request $request, $id)
    {
        $request->validate([
            'admin_remark' => 'required|string|max:255',
        ]);

        $kyc = KycDocument::findOrFail($id);
        $kyc->update([
            'status' => 'rejected',
            'admin_remark' => $request->admin_remark,
        ]);

        // Notify the user
        $kyc->user->notify(new KycStatusUpdated('rejected', $request->admin_remark));

        return redirect()->route('admin.kyc.index')->with('success', 'KYC rejected successfully!');
    }

    public function exportCsv(Request $request)
    {
        // Retrieve filter parameters from the query string
        $search = $request->query('search');
        $status = $request->query('status');

        // Build the query with filters
        $query = KycDocument::with('user');

        // Apply search filter
        if ($search) {
            $query->whereHas('user', function ($query) use ($search) {
                $query->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }

        // Apply status filter
        if ($status) {
            $query->where('status', $status);
        }

        // Retrieve the filtered KYC documents (no pagination)
        $kycDocuments = $query->get();

        // Prepare the CSV output
        $csv = Writer::createFromStream(new SplTempFileObject());
        $csv->insertOne(['ID', 'User Name', 'Email', 'Status', 'Submitted At', 'Last Updated']);

        // Add KYC data rows to the CSV
        foreach ($kycDocuments as $kyc) {
            $csv->insertOne([
                $kyc->id,
                $kyc->user->name,
                $kyc->user->email,
                ucfirst($kyc->status),
                $kyc->created_at->format('d M, Y H:i'),
                $kyc->updated_at->diffForHumans(),
            ]);
        }

        // Return the CSV as a downloadable response
        return response($csv->toString(), 200, [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="kyc_documents.csv"',
        ]);
    }
}
