<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Role;
use App\Models\Trade;
use App\Models\Wallet;
use App\Models\Transaction;
use App\Models\ActivityLog;
use App\Models\Setting;
use App\Models\EmailSetting;
use App\Models\PaymentGatewaySetting;
use App\Services\MT5ManagerService;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail; // <-- Add this line to resolve the error
use App\Mail\UserNotificationMail;   // If using custom mail class
use Illuminate\Support\Facades\Hash;
use App\Models\IBCommissionLevel;
use App\Models\IBRelationship;
use App\Models\IBHierarchy;
use Exception;


class AdminDashboardController extends Controller
{
    protected $mt5Service;

    /**
     * Inject MT5ManagerService into the constructor.
     */
    public function __construct(MT5ManagerService $mt5Service)
    {
        $this->mt5Service = $mt5Service;
    }

    /**
     * Display the admin dashboard.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        Gate::authorize('view-dashboard');

        // Fetch basic statistics
        $totalUsers = User::count();
        $walletBalance = Wallet::sum('balance');
        $activeTrades = Trade::where('status', 'open')->count();
        $newSignups = User::where('created_at', '>=', now()->subDays(7))->count();
        $totalTrades = Trade::count();
        $activeUsers = User::whereHas('trades', fn($query) => $query->where('status', 'open'))->count();

        // Additional report data
        $totalDeposits = Transaction::where('type', 'deposit')->where('status', 'approved')->sum('amount');
        $totalWithdrawals = Transaction::where('type', 'withdrawal')->where('status', 'approved')->sum('amount');
        $activeTradingAccounts = User::whereHas('trades', fn($query) => $query->where('status', 'open'))->count();

        // Recent Transactions
        $recentTransactions = Transaction::with('user')->latest()->take(10)->get();

        // Trades per day (last 30 days)
        $tradesPerDay = Trade::selectRaw('DATE(created_at) as date, count(*) as total')
            ->groupBy('date')
            ->orderBy('date', 'desc')
            ->take(30)
            ->pluck('total', 'date');

        // User signups per day (last 30 days)
        $userSignups = User::selectRaw('DATE(created_at) as date, count(*) as total')
            ->groupBy('date')
            ->orderBy('date', 'desc')
            ->take(30)
            ->pluck('total', 'date');

        // Transaction history (last 30 days)
        $transactionHistory = Transaction::selectRaw('DATE(created_at) as date, type, count(*) as total')
            ->groupBy('date', 'type')
            ->orderBy('date', 'desc')
            ->take(30)
            ->pluck('total', 'date');

        return view('admin.dashboard', [
            'totalUsers' => $totalUsers,
            'walletBalance' => $walletBalance,
            'activeTrades' => $activeTrades,
            'newSignups' => $newSignups,
            'totalTrades' => $totalTrades,
            'activeUsers' => $activeUsers,
            'totalDeposits' => $totalDeposits,
            'totalWithdrawals' => $totalWithdrawals,
            'activeTradingAccounts' => $activeTradingAccounts,
            'recentTransactions' => $recentTransactions,
            'tradesPerDay' => $tradesPerDay,
            'userSignups' => $userSignups,
            'transactionHistory' => $transactionHistory,
        ]);
    }

    /**
     * Manage users.
     *
     * @return \Illuminate\View\View
     */
    public function manageUsers(Request $request)
    {
        // Retrieve query parameters
        $search = $request->query('search');
        $role = $request->query('role');
        $status = $request->query('status'); // Option to filter by user status (active/inactive)

        // Query builder with eager loading for roles
        $query = User::query()->with('roles');

        // Apply search filter for name or email
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }

        // Filter by role if provided
        if ($role) {
            $query->whereHas('roles', function ($q) use ($role) {
                $q->where('name', $role);
            });
        }

        // Filter by user status (active/inactive)
        if ($status) {
            $query->where('is_active', $status === 'active' ? 1 : 0);
        }

        // Paginate the results
        $users = $query->paginate(10);

        // Retrieve all roles for the filter dropdown
        $roles = Role::all();

        // Pass the data to the view
        return view('admin.users', compact('users', 'roles', 'search', 'role', 'status'));
    }

    /**
     * Display activity logs.
     *
     * @return \Illuminate\View\View
     */
    public function activityLogs()
    {
        Gate::authorize('view-admin-activity-logs');

        $logs = ActivityLog::with('user')->latest()->paginate(20);

        return view('admin.activity-logs', compact('logs'));
    }

    /**
     * Display payment gateway settings.
     *
     * @return \Illuminate\View\View
     */
    public function paymentGatewaySettings()
    {
        $gateways = PaymentGatewaySetting::all();

        return view('admin.payment_gateways', compact('gateways'));
    }
    public function updatePaymentGateway(Request $request, $id)
    {
        $gateway = PaymentGatewaySetting::findOrFail($id);

        $request->validate([
            'gateway_name' => 'required|string|max:255',
            'transaction_fee' => 'nullable|numeric|min:0',
            'environment' => 'required|in:sandbox,production',
            'currency' => 'required|string|max:3',
            'min_transaction_amount' => 'nullable|numeric|min:1',
            'success_url' => 'nullable|url',
            'cancel_url' => 'nullable|url',
            'manual_payment_type' => 'nullable|string|max:255',
            'manual_payment_details' => 'nullable|string|max:255',
            'api_key' => 'nullable|string',
            'api_secret' => 'nullable|string',
            'webhook_url' => 'nullable|url',
            'notification_email' => 'nullable|email',
        ]);

        DB::transaction(function () use ($request, $gateway) {
            $gateway->gateway_name = $request->input('gateway_name');
            $gateway->transaction_fee = $request->input('transaction_fee', 0);
            $gateway->environment = $request->input('environment');
            $gateway->currency = $request->input('currency');
            $gateway->gateway_parameters = [
                'min_transaction_amount' => $request->input('min_transaction_amount', 10),
                'success_url' => $request->input('success_url'),
                'cancel_url' => $request->input('cancel_url'),
                'manual_payment_type' => $request->input('manual_payment_type'),
                'manual_payment_details' => $request->input('manual_payment_details'),
                'api_key' => $request->input('api_key'),
                'api_secret' => $request->input('api_secret'),
                'webhook_url' => $request->input('webhook_url'),
                'notification_email' => $request->input('notification_email'),
            ];
            $gateway->is_active = $request->has('is_active');
            $gateway->save();
        });

        return redirect()->route('admin.payment-gateways.index')->with('success', 'Payment gateway updated successfully.');
    }



    /**
     * Update payment gateway settings.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updatePaymentGatewaySettings(Request $request, $id)
    {
        $gateway = PaymentGatewaySetting::findOrFail($id);

        $gateway->is_active = $request->has('is_active');
        $gateway->transaction_fee = $request->input('transaction_fee', 0);
        $gateway->environment = $request->input('environment', 'sandbox');
        $gateway->description = $request->input('description', $gateway->description);

        if ($request->has('gateway_parameters')) {
            $gatewayParameters = $request->input('gateway_parameters');
            if (is_array($gatewayParameters)) {
                $gateway->gateway_parameters = $gatewayParameters;
            }
        }

        $gateway->save();

        return back()->with('success', 'Payment gateway updated successfully.');
        PaymentGatewaySetting::create([
            'gateway_name' => $request->input('gateway_name'),
            'is_active' => $request->has('is_active'),
            'transaction_fee' => 0, // No transaction fee for manual gateways
            'environment' => 'manual', // Set the environment to 'manual' for manual gateways
            'currency' => $request->input('currency'),
            'gateway_parameters' => [
                'min_transaction_amount' => $request->input('min_transaction_amount'),
                'manual_payment_type' => $request->input('manual_payment_type'),
                'manual_payment_details' => $request->input('manual_payment_details'),

            ],
        ]);

        return redirect()->route('admin.payment-gateways')->with('success', 'Manual payment gateway added successfully.');
    }

    /**
     * Show the form for creating a new manual payment gateway.
     *
     * @return \Illuminate\View\View
     */
    public function createManualPaymentGateway()
    {
        return view('admin.create_manual_payment_gateway');
    }

    /**
     * Store a newly created manual payment gateway in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function storeManualPaymentGateway(Request $request)
    {
        // Validate request data
        $validated = $request->validate([
            'gateway_name' => 'required|string|max:255',
            'manual_payment_type' => 'required|string|max:255',
            'manual_payment_details' => 'required|string|max:500',
            'currency' => 'required|string|max:3',
            'min_transaction_amount' => 'required|numeric|min:0',
            'is_active' => 'nullable|boolean',
        ]);

        // Store manual payment gateway
        $gateway = PaymentGatewaySetting::create([
            'gateway_name' => $validated['gateway_name'],
            'manual_payment_type' => $validated['manual_payment_type'],
            'manual_payment_details' => $validated['manual_payment_details'],
            'currency' => $validated['currency'],
            'min_transaction_amount' => $validated['min_transaction_amount'],
            'is_active' => $request->has('is_active') ? 1 : 0,
            'is_manual' => 1, // Mark it as manual payment gateway
        ]);

        return redirect()->route('admin.payment.gateways')->with('success', 'Manual payment gateway added successfully.');
    }

    /**
     * Create a new payment gateway.
     *
     * @return \Illuminate\View\View
     */
    public function createPaymentGateway()
    {
        return view('admin.create_payment_gateway');
    }

    /**
     * Store a new payment gateway.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function storePaymentGateway(Request $request)
    {
        // Validate input data
        $validatedData = $request->validate([
            'gateway_name' => 'required|string|max:255',
            'is_active' => 'nullable|boolean',
            'manual_payment_type' => 'required|string|max:255',
            'manual_payment_details' => 'required|string|max:1000',
            'currency' => 'required|string|max:3',
            'min_transaction_amount' => 'required|numeric|min:0',
            'is_active' => 'nullable|boolean',
        ]);

        PaymentGatewaySetting::create([
            'gateway_name' => $validatedData['gateway_name'],
            'is_active' => $request->has('is_active') ? 1 : 0,
            'manual_payment_type' => $validatedData['manual_payment_type'],
            'manual_payment_details' => $validatedData['manual_payment_details'],
            'currency' => $validatedData['currency'],
            'min_transaction_amount' => $validatedData['min_transaction_amount'],
            'is_manual' => true, // Setting 'is_manual' column to true

        ]);

        return redirect()->route('admin.payment.gateways')->with('success', 'Payment gateway created successfully.');
    }
    /**
     * Delete a payment gateway.
     *
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function deletePaymentGateway($id)
    {
        $gateway = PaymentGatewaySetting::findOrFail($id);
        $gateway->delete();

        return redirect()->route('admin.payment.gateways')->with('success', 'Payment gateway deleted successfully.');
    }

    /**
     * Approve a transaction.
     *
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function approveTransaction($id)
    {
        $transaction = Transaction::find($id);

        if (!$transaction) {
            return redirect()->back()->with('error', 'Transaction not found.');
        }

        $transaction->status = 'approved';
        $transaction->save();

        if ($transaction->type === 'deposit') {
            $wallet = $transaction->user->wallet;
            $wallet->balance += $transaction->amount;
            $wallet->save();
        }

        return redirect()->back()->with('success', 'Transaction approved successfully!');
    }

    /**
     * View all trades for a specific user.
     *
     * @param int $userId
     * @return \Illuminate\View\View|\Illuminate\Http\RedirectResponse
     */
    public function viewTrades($userId)
    {
        $user = User::findOrFail($userId);

        if (!$user->mt5_account_id) {
            return redirect()->back()->with('error', 'User does not have an MT5 account linked.');
        }

        try {
            $response = $this->mt5Service->getTrades($user->mt5_account_id);

            if ($response['status'] === 'success') {
                return view('admin.trades', ['trades' => $response['data']]);
            }

            return redirect()->back()->with('error', $response['message'] ?? 'Failed to fetch trades.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Error fetching trades: ' . $e->getMessage());
        }
    }




    /**
     * Edit user details.
     *
     * @param int $id
     * @return \Illuminate\View\View
     */
    public function editUser($id)
    {
        $user = User::findOrFail($id);
        $roles = Role::all(); // Assuming you have a Role model

        return view('admin.users.edit', compact('user', 'roles'));
    }


    /**
     * Delete a user.
     *
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function deleteUser($id)
    {
        Gate::authorize('delete-users');

        $user = User::findOrFail($id);
        $user->delete();

        return redirect()->route('admin.users')->with('success', 'User deleted successfully.');
    }

    public function listMT5Accounts()
    {
        $accounts = $this->mt5Service->getAllAccounts();

        if ($accounts['status'] === 'error') {
            session()->flash('warning', $accounts['message']);
            $accounts['data'] = []; // Return empty data if MT5 is not connected
        }

        return view('admin.mt5_accounts', ['accounts' => $accounts['data']]);
    }


    public function listTrades()
    {
        $trades = $this->mt5Service->getAllTrades();
        return view('admin.mt5_trades', [
            'trades' => $trades['status'] === 'success' ? $trades['data'] : [],
        ]);
    }

    public function updateUser(Request $request, $id)
    {
        $user = User::findOrFail($id);

        // Validate the incoming request
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $id,
            'role' => 'nullable|string',
            'is_active' => 'nullable|boolean',
        ]);

        // Update user attributes
        $user->name = $request->input('name');
        $user->email = $request->input('email');
        $user->is_active = $request->has('is_active') ? 1 : 0;

        // Assign role to the user
        if ($request->filled('role')) {
            // Detach existing roles before assigning the new role
            $user->roles()->detach();

            // Assign the new role
            $user->assignRole($request->input('role'));
        }

        // Save user updates
        $user->save();

        return redirect()->route('admin.users')->with('success', 'User updated successfully.');
    }
    //toggle active users

    public function toggleUserStatus(Request $request, $id)
    {
        // Authorize the action (optional but recommended)
        Gate::authorize('manage-users');

        // Validate the request data
        $validated = $request->validate([
            'is_active' => 'required|boolean',
        ]);

        // Find the user by ID
        $user = User::findOrFail($id);

        // Update the user's active status
        $user->is_active = $validated['is_active'];
        $user->save();

        return response()->json(['success' => true, 'message' => 'User status updated successfully.']);
    }


    public function updateUserRole(Request $request)
    {
        Gate::authorize('manage-users');
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'roles' => 'array|exists:roles,id',
        ]);

        $user = User::findOrFail($request->user_id);
        $user->roles()->sync($request->roles);

        return redirect()->route('admin.users')->with('success', 'User roles updated successfully.');
    }


    public function viewReports()
    {
        // Fetch data for the reports
        $totalUsers = User::count();
        $totalTrades = Trade::count();
        $totalTransactions = Transaction::count();
        $totalBalance = Wallet::sum('balance');

        // Calculate total deposits
        $totalDeposits = Transaction::where('type', 'deposit')->where('status', 'approved')->sum('amount');

        // Calculate total withdrawals
        $totalWithdrawals = Transaction::where('type', 'withdrawal')->where('status', 'approved')->sum('amount');

        // Fetch trades per day for the last 30 days
        $tradesPerDay = Trade::selectRaw('DATE(created_at) as date, count(*) as total')
            ->groupBy('date')
            ->orderBy('date', 'desc')
            ->take(30)
            ->pluck('total', 'date');

        // Fetch transaction history for the last 30 days
        $transactionHistory = Transaction::selectRaw('DATE(created_at) as date, type, sum(amount) as total')
            ->groupBy('date', 'type')
            ->orderBy('date', 'desc')
            ->take(30)
            ->get()
            ->groupBy('date')
            ->mapWithKeys(function ($items, $date) {
                return [$date => $items->sum('total')];
            });
        // Fetch user signups per day for the last 30 days
        $userSignups = User::selectRaw('DATE(created_at) as date, count(*) as total')
            ->groupBy('date')
            ->orderBy('date', 'desc')
            ->take(30)
            ->pluck('total', 'date');

        // Fetch top 5 users by deposits
        $topUsersByDeposits = Transaction::where('type', 'deposit')
            ->where('status', 'approved')
            ->select('user_id', DB::raw('SUM(amount) as total_amount'))
            ->groupBy('user_id')
            ->orderBy('total_amount', 'desc')
            ->take(5)
            ->pluck('total_amount', 'user_id');

        // Fetch top 5 users by withdrawals
        $topUsersByWithdrawals = Transaction::where('type', 'withdrawal')
            ->where('status', 'approved')
            ->select('user_id', DB::raw('SUM(amount) as total_amount'))
            ->groupBy('user_id')
            ->orderBy('total_amount', 'desc')
            ->take(5)
            ->pluck('total_amount', 'user_id');

        // Monthly deposits vs withdrawals comparison
        $monthlyDepositWithdrawalComparison = Transaction::selectRaw('DATE_FORMAT(created_at, "%Y-%m") as month, type, SUM(amount) as total_amount')
            ->groupBy('month', 'type')
            ->orderBy('month', 'asc')
            ->get()
            ->groupBy('month');

        // Calculate revenue analysis (assuming revenue is total deposits minus withdrawals)
        $revenueAnalysis = Transaction::selectRaw('DATE(created_at) as date, sum(amount) as total')
            ->where('type', 'revenue')
            ->groupBy('date')
            ->orderBy('date')
            ->get()
            ->pluck('total', 'date');

        // Users with active wallets (wallet balance greater than zero)
        $usersWithActiveWallets = Wallet::where('balance', '>', 0)
            ->distinct('user_id')
            ->count();


        $usersWithActiveTrades = User::whereHas('trades', function ($query) {
            $query->where('status', 'active');
        })->count();

        $pendingWithdrawals = Transaction::where('type', 'withdrawal')->where('status', 'pending')->sum('amount');

        $newUsersLastMonth = User::where('created_at', '>=', now()->subDays(30))->count();


        // Calculate average trade size
        $averageTradeSize = Trade::avg('amount');

        // Fetch recent transactions
        $recentTransactions = Transaction::with('user')->latest()->take(10)->get();

        // Fetch active trading accounts using user_id instead of account_id
        $activeTradingAccounts = Trade::where('status', 'open')->distinct('user_id')->count();

        return view('admin.reports', [
            'totalUsers' => $totalUsers,
            'totalTrades' => $totalTrades,
            'totalTransactions' => $totalTransactions,
            'totalBalance' => $totalBalance,
            'totalDeposits' => $totalDeposits,
            'totalWithdrawals' => $totalWithdrawals,
            'activeTradingAccounts' => $activeTradingAccounts,
            'recentTransactions' => $recentTransactions,
            'tradesPerDay' => $tradesPerDay, // Add this line to pass the $tradesPerDay data
            'userSignups' => $userSignups, // Add this line to pass the $userSignups data
            'transactionHistory' => $transactionHistory,
            'averageTradeSize' => $averageTradeSize,
            'topUsersByDeposits' => $topUsersByDeposits,
            'topUsersByWithdrawals' => $topUsersByWithdrawals,
            'monthlyDepositWithdrawalComparison' => $monthlyDepositWithdrawalComparison,
            'revenueAnalysis' => $revenueAnalysis,
            'pendingWithdrawals' => $pendingWithdrawals,
            'usersWithActiveTrades' => $usersWithActiveTrades,
            'newUsersLastMonth' => $newUsersLastMonth,
            'usersWithActiveWallets' => $usersWithActiveWallets,
        ]);
    }


    public function commissionLevels()
    {
        $commissionLevels = IBCommissionLevel::all();
        return view('admin.commission-levels.index', compact('commissionLevels'));
    }

    public function updateCommissionLevel(Request $request, $id)
    {
        $request->validate([
            'commission_percentage' => 'required|numeric|min:0|max:100',
        ]);

        $commissionLevel = IBCommissionLevel::findOrFail($id);
        $commissionLevel->commission_percentage = $request->commission_percentage;
        $commissionLevel->save();

        return redirect()->route('admin.commission.levels')->with('success', 'Commission level updated successfully');
    }

    /**
     * Show the form for creating a new user.
     *
     * @return \Illuminate\View\View
     */
    public function createUser()
    {
        return view('admin.create-user');
    }

    /**
     * Store a newly created user in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function storeUser(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8|confirmed',
            'is_ib' => 'nullable|boolean',
            'is_active' => 'required|boolean',
        ]);

        // Create a new user
        User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'is_ib' => $request->is_ib ?? false,
            'is_active' => $request->is_active,
        ]);

        return redirect()->route('admin.user.management')->with('success', 'User created successfully.');
    }

    //email setting

    public function showEmailSettings()
    {
        $emailSettings = EmailSetting::first();
        return view('admin.email_settings', compact('emailSettings'));
    }

    public function updateEmailSettings(Request $request)
    {
        $request->validate([
            'smtp_host' => 'required|string',
            'smtp_port' => 'required|integer',
            'smtp_username' => 'required|string',
            'smtp_password' => 'required|string',
            'smtp_encryption' => 'nullable|string',
            'from_email' => 'required|email',
            'from_name' => 'required|string',
        ]);

        EmailSetting::updateOrCreate(['id' => 1], $request->all());

        return redirect()->route('admin.email-settings')->with('success', 'Email settings updated successfully.');
    }

    /**
     * Display MT5 settings and update functionality.
     */
    public function viewMT5Settings()
    {
        $realSettings = DB::table('mt5_settings')->where('server_type', 'real')->first();
        $demoSettings = DB::table('mt5_settings')->where('server_type', 'demo')->first();

        return view('admin.mt5_settings', compact('realSettings', 'demoSettings'));
    }

    public function updateMT5Settings(Request $request)
    {
        $request->validate([
            'type' => 'required|in:real,demo',
            'server' => 'required|ip',
            'port' => 'required|integer|min:1|max:65535',
            'manager_login' => 'required|string|max:255',
            'manager_password' => 'required|string|max:255',
        ]);

        DB::table('mt5_settings')->updateOrInsert(
            ['server_type' => $request->type],
            [
                'server' => $request->server,
                'port' => $request->port,
                'manager_login' => $request->manager_login,
                'manager_password' => $request->manager_password,
                'updated_at' => now(),
            ]
        );

        return redirect()->route('admin.mt5.settings')->with('success', ucfirst($request->type) . ' MT5 settings updated successfully!');
    }


    public function showSendEmailForm($id)
    {
        $user = User::findOrFail($id);
        return view('admin.send_email', compact('user'));
    }

    public function sendEmailToUser(Request $request, $id)
    {
        $user = User::findOrFail($id);

        $request->validate([
            'subject' => 'required|string|max:255',
            'message' => 'required|string',
        ]);

        $content = [
            'subject' => $request->subject,
            'title' => 'Message from Admin',
            'message' => $request->message,
        ];

        Mail::to($user->email)->send(new UserNotificationMail($content));

        return redirect()->route('admin.users')->with('success', 'Email sent successfully.');
    }


    public function transferFunds(Request $request)
    {
        $request->validate([
            'login' => 'required|string',
            'amount' => 'required|numeric|min:1',
            'comment' => 'nullable|string',
            'action' => 'required|in:deposit,withdrawal', // Action field is mandatory
        ]);

        $login = $request->login;
        $amount = $request->amount;
        $comment = $request->comment ?? 'Fund Transfer';
        $action = $request->action;

        try {
            $response = $this->mt5Service->transferFunds($login, $amount, $comment, $action);

            if ($response['status'] === 'success') {
                return redirect()->back()->with('success', ucfirst($action) . ' completed successfully.');
            }

            return redirect()->back()->with('error', $response['message'] ?? 'Transfer failed.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }
}
